
from django.db import models
from django.db.models.base import ModelBase
from django.utils.translation import get_language
from mptt.models import MPTTModel, MPTTModelBase


def models_set(models_list):
    class AlwaysContainsSet(set):
        def __contains__(self, value):
            return True

    set_class = AlwaysContainsSet if '__all__' in models_list else set
    return set_class(models_list)


class IntranetManager(models.Manager):
    def active(self):
        return super(IntranetManager, self).filter(intranet_status__gt=0)

    def intranet_filter(self, *args, **kwargs):
        kwargs['intranet_status'] = 1
        return models.Manager.filter(self, *args, **kwargs)


class I18nBaseModel(ModelBase):
    def __new__(cls, *args, **kwargs):
        model = super(I18nBaseModel, cls).__new__(cls, *args, **kwargs)
        model._meta._i18n_fields = {}
        for f in model._meta.fields:
            if f.name[-3:] == '_en':
                model._meta._i18n_fields[f.name[:-3]] = True
        return model


class I18nMpttBaseModel(MPTTModelBase, I18nBaseModel):
    pass


def get_i_field(model_dict, field, prefix=''):
    en_field = prefix + field + '_en'
    nl_field = prefix + 'native_lang'
    if en_field in model_dict and nl_field in model_dict:
        lang = get_language()
        pos = lang.find('-')
        if pos > 0:
            lang = lang[:pos]
        if lang != model_dict[nl_field]:
            return model_dict[en_field]
    return model_dict[prefix + field]


class I18nModel(models.Model, metaclass=I18nBaseModel):
    """Основное правило: если у сущности есть интернационализованные поля
    (например office.name и office.name_en), то для правильной работы в поле
    native_lang должно быть записано значение языка, на котором записана
    информация в первом поле (для офиса office.name). Таким образом, если
    локаль человека не совпадает с native_lang для сущности, на которую он
    смотрит, то язык для него переключится на интернациональный английский.

    """

    _i18n_fields = {}

    native_lang = models.CharField(max_length=2, default='', blank=True)

    def __getattr__(self, attr_name):

        if not attr_name.startswith('i_'):
            # Если атрибут не найден по цепочке наследования и
            # не начинается с i_, значит его нигде нет. Обработка ситуации,
            # что у родительского класса есть __getattr__ с какой-то
            # дополнительной логикой все сильно усложнит и кажется маловероятной.
            raise AttributeError(attr_name)

        i18n_fields = self._meta._i18n_fields
        native_lang = self.native_lang

        attr_name = attr_name[2:]
        if attr_name in i18n_fields and native_lang:
            lang = get_language()
            if lang:
                lang = lang.split('-')[0]

            # если выбранный язык не совпадает с native_lang, то
            # поле attr_name следует перевести на международный (английский).
            if lang != native_lang:
                translated = getattr(self, attr_name + '_en')
                if translated:
                    return translated

        return getattr(self, attr_name)

    class Meta:
        abstract = True


class SoftDeletedModel(models.Model):
    intranet_status = models.IntegerField(default=1)

    objects = IntranetManager()
    all = models.Manager()

    class Meta:
        abstract = True


class IntranetModel(I18nModel, SoftDeletedModel):
    created_at = models.DateTimeField(editable=False)
    modified_at = models.DateTimeField(editable=False)

    class Meta:
        abstract = True
        app_label = 'intranet'


from mptt.managers import TreeManager


class IntranetMpttManager(IntranetManager, TreeManager):
    pass


class I18nMpttModel(MPTTModel, I18nModel, metaclass=I18nMpttBaseModel):
    class Meta(MPTTModel.Meta):
        abstract = True
        app_label = 'intranet'

    tree = IntranetMpttManager()


class IntranetMpttModel(MPTTModel, IntranetModel, metaclass=I18nMpttBaseModel):
    class Meta(MPTTModel.Meta):
        abstract = True
        app_label = 'intranet'

    tree = IntranetMpttManager()
