from django.contrib.auth import get_user_model
from django.contrib.auth.models import AnonymousUser
from django.db import models

from wiki.intranet.models.base import IntranetManager, IntranetMpttModel, IntranetMpttManager
from wiki.intranet.models.consts import GROUP_TYPE_CHOICES
from wiki.intranet.models.exceptions import GroupNotFound, GroupFoundMultiple
from wiki.intranet.models.staff import Staff


class GroupManager(IntranetManager):
    intranet_root_kwargs = {'url': '__wiki__', 'level': 0}

    def get_intranet_root(self):
        return self.get(**self.intranet_root_kwargs)

    def get_admins(self):
        return self.get(name='Admins', parent__url='__wiki__', parent__level=0)

    departments_root_kwargs = {'url': '__departments__', 'level': 0}

    def get_departments_root(self):
        return self.get(**self.departments_root_kwargs)

    services_root_kwargs = {'url': '__services__', 'level': 0}

    def get_services_root(self):
        return self.get(**self.services_root_kwargs)

    intra_services_root_kwargs = {'url': '__intranet_services__', 'level': 0}

    def get_intra_services_root(self):
        return self.get(**self.intra_services_root_kwargs)

    def is_admin(self, user):
        if not user or isinstance(user, AnonymousUser) or not Staff:
            return False
        kwargs = {'group__name': 'Admins', 'group__parent__url': '__wiki__', 'group__parent__level': 0}
        if isinstance(user, get_user_model()):
            if not user.is_authenticated:
                return False
            kwargs.update({'staff__login': user.username})
        elif isinstance(user, str):
            kwargs.update({'staff__login': user})
        elif isinstance(user, Staff):
            kwargs.update({'staff': user})
        return GroupMembership.objects.filter(**kwargs).exists()

    def get_type_queryset(self, type_):
        if type_ == 'intra':
            type_ = 'wiki'

        available = [key for __, key in GROUP_TYPE_CHOICES.choices()]

        if type_ not in available:
            raise ValueError('Type must be one of: %s' % ', '.join(available))

        return self.get_query_set().filter(type=GROUP_TYPE_CHOICES.get_key(type_))

    def search_exactly(self, query, typ):
        """
        Search exact fit group by query and type
        Used in interfaces as defense of groups autocomlete from incorrect input
        """
        qs = self.get_type_queryset(typ).filter(intranet_status__gt=0)

        try:
            try:
                r = qs.get(name=query)
            except Group.DoesNotExist:
                try:
                    r = qs.get(url=query)
                except Group.DoesNotExist:
                    raise GroupNotFound('Group "%s" has not been found' % query)
        except Group.MultipleObjectsReturned:
            raise GroupFoundMultiple('There are several groups named "%s"' % query)
        return r


class Group(IntranetMpttModel):
    class MPTTMeta:
        order_insertion_by = ['position']

    class Meta(IntranetMpttModel.Meta):
        db_table = 'intranet_group'

    tree = IntranetMpttManager()
    objects = GroupManager()

    OPTION = 808  # evil hidden group
    department = models.OneToOneField('intranet.Department', null=True, blank=True, on_delete=models.CASCADE)

    service_id = models.PositiveIntegerField(null=True, blank=True)
    role_scope_id = models.CharField(max_length=127, null=True, blank=True)

    parent = models.ForeignKey('self', null=True, blank=True, related_name='children', on_delete=models.CASCADE)
    name = models.CharField(max_length=127, db_index=True)
    url = models.CharField(max_length=255, unique=True, db_index=True)
    code = models.CharField(max_length=127, db_index=True, default='')
    description = models.TextField()
    position = models.IntegerField(null=True, default=0)

    type = models.PositiveSmallIntegerField(
        choices=GROUP_TYPE_CHOICES.choices(),
        default=GROUP_TYPE_CHOICES.WIKI,
        blank=False,
    )

    members = models.ManyToManyField('Staff', through='GroupMembership', related_name='in_groups')
    # денормализация: число внешних сотрудников в группе.
    externals_count = models.PositiveIntegerField(default=0)
    yandex_count = models.PositiveIntegerField(default=0)
    yamoney_count = models.PositiveIntegerField(default=0)

    def get_public_group_id(self):
        return str(self.pk)

    def get_all_members(self, include_dismissed=False):
        """
        Return all members of this group and all its subgroups as
        QuerySet object. Dismissed employees are ignored.
        """

        staff_ids = GroupMembership.objects.filter(
            group__tree_id=self.tree_id, group__lft__gte=self.lft, group__rght__lte=self.rght
        ).values('staff_id')

        members = Staff.objects.filter(id__in=staff_ids)

        if not include_dismissed:
            members = members.filter(is_dismissed=False)

        return members

    @property
    def all_members(self):
        """
        Property stands for backward compatibility. New code should use
        `get_all_members` method that returns QuerySet instead of list
        and thus can be further tuned
        """
        return list(self.get_all_members())

    @property
    def is_intranet_service(self):
        url = Group.objects.intra_services_root_kwargs['url']
        ancs = self.get_ancestors()
        return self.url == url or (len(ancs) and ancs[0].url == url)

    @property
    def is_intranet(self):
        return self.is_wiki

    @property
    def is_wiki(self):
        return self.type == GROUP_TYPE_CHOICES.WIKI

    @property
    def is_service(self):
        return self.type == GROUP_TYPE_CHOICES.SERVICE

    @property
    def is_servicerole(self):
        return self.type == GROUP_TYPE_CHOICES.SERVICEROLE

    @property
    def is_department(self):
        return self.type == GROUP_TYPE_CHOICES.DEPARTMENT

    def __str__(self):
        return self.name


class GroupMembership(models.Model):
    staff = models.ForeignKey(Staff, on_delete=models.CASCADE)
    group = models.ForeignKey(Group, on_delete=models.CASCADE)
    joined_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return str(self.staff) + ' in ' + str(self.group)

    class Meta:
        db_table = 'intranet_groupmembership'
        unique_together = (('staff', 'group'),)
        app_label = 'intranet'
