
from functools import reduce
from operator import or_

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import models
from django.db.models import Q

from wiki.intranet.models.base import IntranetManager, IntranetModel
from wiki.intranet.models.consts import (
    AFFILIATION,
    DOMAIN,
    EDU_DIRECTION,
    EDU_STATUS,
    EMPLOYMENT,
    FAMILY_STATUS,
    GENDER,
    LANG,
)
from wiki.intranet.models.exceptions import UserFoundMultiple, UserNotFound
from wiki.intranet.models.inflections import StaffInflection
from wiki.legacy.json import loads
from wiki.legacy.orm import JSONField

user_model = settings.AUTH_USER_MODEL


class StaffManager(IntranetManager):
    def get_by_user(self, user):
        if user and user.is_authenticated:
            try:
                return self.get(user=user)
            except Staff.DoesNotExist:
                pass
        return None

    def search_exactly(self, query):
        """
        Search exact fit user by query (login, name or surname)
        Used in interfaces as defense of user's autocomlete from incorrect input
        """

        qs = self.filter(is_dismissed=False)

        try:
            if len(query.split(' ')) == 2:
                t = query.split(' ')
                try:
                    r = qs.get(first_name=t[0], last_name=t[1])
                except Staff.DoesNotExist:
                    try:
                        r = qs.get(first_name=t[1], last_name=t[0])
                    except Staff.DoesNotExist:
                        raise UserNotFound('User "%s" has not been found' % query)
            else:
                try:
                    r = qs.get(login=query)
                except Staff.DoesNotExist:
                    try:
                        r = qs.get(last_name=query)
                    except Staff.DoesNotExist:
                        try:
                            r = qs.get(first_name=query)
                        except Staff.DoesNotExist:
                            raise UserNotFound('User "%s" has not been found' % query)
        except Staff.MultipleObjectsReturned:
            raise UserFoundMultiple('There are several users named "%s"' % query)
        return r


class Staff(IntranetModel):
    user = models.OneToOneField(user_model, null=True, on_delete=models.CASCADE)
    from_staff_id = models.PositiveIntegerField(db_index=True, default=0)
    # В терминах blackbox и passport это поле называется display_name.
    # Оно содержит ненормализованный логин, каким его видит пользователь, с точками.
    login = models.CharField(max_length=50, db_index=True, default='')
    # Нормализованный логин, который не содержит точек.
    normal_login = models.CharField(max_length=50, db_index=True, default='')
    login_mail = models.CharField(max_length=50, db_index=True, default='')

    domain = models.CharField(max_length=1, choices=DOMAIN.choices(), default='')
    login_ld = models.CharField(max_length=50, db_index=True, unique=True)
    uid = models.CharField(max_length=16, db_index=True, null=True, unique=True)

    first_name = models.CharField(max_length=50, default='', db_index=True)
    first_name_en = models.CharField(max_length=50, default='', db_index=True)
    middle_name = models.CharField(max_length=50, default='')
    hide_middle_name = models.BooleanField(default=True)
    last_name = models.CharField(max_length=100, default='', db_index=True)
    last_name_en = models.CharField(max_length=100, default='', db_index=True)
    en_name = models.CharField(max_length=255, default='')
    tz = models.CharField(max_length=30, default='')

    # https://st.yandex-team.ru/STAFF-3520
    is_homeworker = models.BooleanField(default=False)
    is_robot = models.BooleanField(default=False)
    affiliation = models.CharField(max_length=32, db_index=True, choices=AFFILIATION.choices())

    # https://st.yandex-team.ru/ML-1291
    has_exchange = models.BooleanField(default=False)
    last_offer_id = models.IntegerField(null=True)
    is_dismissed = models.BooleanField(default=False)
    work_email = models.CharField(max_length=100, default='', db_index=True)
    wiki_name = models.CharField(max_length=100, db_index=True, default='')
    lang_ui = models.CharField(max_length=2, choices=LANG.choices(), default='')
    lang_content = models.CharField(max_length=255, default='')
    service_profile = JSONField(max_length=4000, default='{}')
    gender = models.CharField(max_length=1, choices=GENDER.choices(), default='')

    guid = models.CharField(max_length=47, null=True, default='')

    @property
    def is_active(self):
        return not self.is_dismissed

    @property
    def email(self):
        return self.work_email

    def _get_langs(self):
        return [x for x in self.lang_content.split('|') if x]

    def _set_langs(self, langs):
        self.lang_content = '|'.join([x for x in langs if x])

    langs = property(_get_langs, _set_langs)

    @property
    def profile(self):
        try:
            return loads(self.service_profile) if self.service_profile else {}
        except ValueError:
            return {}

    @property
    def username(self):
        return self.login

    def get_full_name(self):
        if self.i_first_name and self.i_first_name == self.i_last_name and ' ' in self.i_last_name:
            name = self.i_last_name
        else:
            name = ('%s %s' % (self.i_first_name, self.i_last_name)).strip()

        return name if name else self.login

    __str__ = get_full_name

    # < интранетовские расширения >
    # интересно, где что используется

    if settings.IS_INTRANET:
        birthday = models.DateField(null=True)
        hide_birthday_year = models.BooleanField(default=False)
        family_status = models.CharField(max_length=1, choices=FAMILY_STATUS.choices(), default='')
        children = models.PositiveSmallIntegerField(null=True)
        car = models.CharField(max_length=100, default='')
        car_num = models.CharField(max_length=100, default='')
        address = models.CharField(max_length=255, default='')
        address_en = models.CharField(max_length=255, default='')
        edu_status = models.CharField(max_length=1, choices=EDU_STATUS.choices(), default='')
        edu_direction = models.CharField(max_length=1, choices=EDU_DIRECTION.choices(), default='')
        edu_place = models.CharField(max_length=255, default='')
        edu_place_en = models.CharField(max_length=255, default='')
        edu_date = models.DateField(null=True)
        department = models.ForeignKey('intranet.Department', null=True, on_delete=models.CASCADE)
        is_big_boss = models.BooleanField(default=False)
        office = models.ForeignKey('intranet.Office', null=True, on_delete=models.CASCADE)
        desk_id = models.IntegerField(null=True)
        join_at = models.DateField(null=True)
        quit_at = models.DateField(null=True)
        position = models.CharField(max_length=150, default='')
        position_en = models.CharField(max_length=150, default='')

        employment = models.CharField(max_length=1, choices=EMPLOYMENT.choices(), default='')

        work_phone = models.PositiveIntegerField(null=True, db_index=True)
        mobile_phone = models.CharField(max_length=100, default='')

        @property
        def departments(self):
            if self.department is None:
                return []
            lst = list(self.department.get_ancestors(False))
            lst.append(self.department)
            return lst

        @property
        def is_external_employee(self):
            if self.department is None:
                return True
            return not self.department.url.startswith('yandex')

        def get_all_groups(self, fields=None):

            from wiki.intranet.models.intranet_extensions import Group

            my_groups_qs = self.in_groups.filter(intranet_status=1)
            my_groups_qs = my_groups_qs.values('lft', 'rght', 'tree_id')

            q_gen = (Q(lft__lte=g['lft'], rght__gte=g['rght'], tree_id=g['tree_id']) for g in my_groups_qs)

            q = reduce(or_, q_gen, Q())

            # if user isn't in any group
            if not q:
                return []

            groups_qs = Group.objects.filter(q).filter(level__gt=0, intranet_status=1)

            if fields:
                groups_qs = groups_qs.values(*fields)

            return list(groups_qs)

        @property
        def all_groups(self):
            return self.get_all_groups()

    # < /интранетовские расширения >

    @property
    def django_user(self):
        if not hasattr(self, '_django_user'):
            self._django_user, is_created = get_user_model().objects.get_or_create(
                username=self.login, defaults={'email': self.get_email()}
            )
            if is_created:
                self.user = self._django_user
                self.save()
        return self._django_user

    def get_email(self):
        if hasattr(self, 'work_email'):
            return self.work_email if self.work_email is not None else ''
        elif hasattr(self, 'domain') and hasattr(self, 'login_mail'):
            return '%s@%s' % (self.login_mail, self.get_domain_display())
        return '%s@yandex-team.ru' % self.login

    @property
    def inflections(self):
        return StaffInflection(staff=self)

    def save(self, **kwargs):
        """extends parent save

        Сохранить модель, записать в self.normal_login значение из self.login заменив точки на минусы
        """
        if self.login is not None:
            self.normal_login = self.login.replace('.', '-')
        else:
            self.normal_login = ''
        super(Staff, self).save()

    objects = StaffManager()

    class Meta(IntranetModel.Meta):
        db_table = 'intranet_staff'
