import itertools
import urllib.parse

from django import template
from django.conf import settings
from django.utils import translation
from django.utils.html import escape
from django.utils.safestring import mark_safe

from django.utils.http import urlencode
from django.utils.encoding import smart_str

from wiki.legacy import slave
from wiki.intranet.models import Staff

STAFF_LINK_TEMPLATE = (
    '<lego:b-user url="http://staff.yandex-team.ru/%(login)s" ' 'card="%(login)s" %(attrs)s>%(name)s</lego:b-user>'
)

PLAIN_STAFF_LINK_TEMPLATE = '<a href="http://staff.yandex-team.ru/%(login)s" ' '>%(name)s</a>'

USERPIC_TEMPLATE = (
    '''
    <span class="%(css_class)s">
        <a href="//staff.yandex-team.ru/%(username)s">
            <img src="//center.yandex-team.ru/user/avatar/'''
    '''%(username)s/%(size)s/%(tail)s" alt="%(full_name)s" />
                           </a>
                       </span>'''
)

register = template.Library()


def _lang():
    lang = translation.get_language() or settings.LANGUAGE_CODE
    return lang.split('-')[0]


@register.simple_tag
def replace_in_query(query, key, newvalue):
    splitted = urllib.parse.urlsplit(smart_str(query))
    parsed_query = urllib.parse.parse_qs(splitted[-2])
    parsed_query[key] = [newvalue]
    newsplitted = list(splitted)
    newsplitted[-2] = urlencode(parsed_query, doseq=1)
    return mark_safe(urllib.parse.urlunsplit(newsplitted))


@register.filter(name='email_link')
def email_link(email, name=None):
    if not name:
        name = email = escape(email)
    else:
        name, email = escape(name), escape(email)
    return mark_safe('<a href="mailto:%s" title="%s">%s</a>' % (email, email, name))


@register.filter
def split_in_chunks(lst, nchunks):
    in_chunk, mod = divmod(len(lst), nchunks)
    if mod > 0:
        in_chunk += 1

    itr = iter(lst)
    for nchunk in range(nchunks):
        yield itertools.islice(itr, in_chunk)


@register.tag
def setvar(parser, token):
    try:
        # split_contents() knows not to split quoted strings.
        tag_name, varname, value = token.split_contents()
    except ValueError:
        raise template.TemplateSyntaxError('%r tag requires exactly two arguments' % token.contents.split()[0])

    if not (varname[0] == varname[-1] and varname[0] in ('"', "'")):
        raise template.TemplateSyntaxError("%r tag's first argument should be in quotes" % tag_name)
    return SetVarNode(varname[1:-1], value)


@register.filter
def get(mapping, attrname):
    return mapping.get(attrname)


class SetVarNode(template.Node):
    def __init__(self, varname, value):
        self.varname = varname
        self.value = template.Variable(value)

    def render(self, context):
        try:
            actual_value = self.value.resolve(context)
            context[self.varname] = actual_value
        except template.VariableDoesNotExist:
            pass

        return ''


@register.filter
def sub(value, arg):
    """Subtracts the arg from the value."""
    try:
        return int(value) - int(arg)
    except (ValueError, TypeError):
        try:
            return value - arg
        except Exception:
            return ''


def _staff_link(staff, inflection=None, is_plain=False):
    if isinstance(staff, str) and not staff.isdigit():
        # interpret as login
        # NOTE: don't go to DB so we don't know if one is dismissed
        name = login = staff
        attrs = ''
    else:
        if isinstance(staff, (int, str)):
            # interpret as primary key (NOTE: and go to db)
            staff = Staff.objects.get(pk=staff)

        if inflection:
            name = staff.inflections.inflect(inflection)
        else:
            name = staff.get_full_name()
        login = staff.login

        if not is_plain and staff.is_dismissed:
            attrs = 'lego:type="fired"'
        else:
            attrs = ''

    template = PLAIN_STAFF_LINK_TEMPLATE if is_plain else STAFF_LINK_TEMPLATE

    return mark_safe(
        template
        % {
            'login': login,
            'name': name,
            'attrs': attrs,
        }
    )


@register.filter
def staff_link(staff, inflection=None):
    return _staff_link(staff, inflection=inflection, is_plain=False)


@register.filter
def plain_staff_link(staff, inflection=None):
    return _staff_link(staff, inflection=inflection, is_plain=True)


@register.simple_tag
def staff_userpic(staff, size=64, css_class='userpic', is_square=True):
    context = {
        'username': staff.login,
        'full_name': staff.get_full_name().replace('"', ''),  # escape "
        'css_class': css_class,
        'size': size,
        'tail': 'square/' if is_square else '',
    }
    result = USERPIC_TEMPLATE % context
    return mark_safe(result)


@register.inclusion_tag('intranet/templatetags/is_slave_message.html', takes_context=True)
def is_slave_message(context):
    request = context.get('request', None)
    if request:
        try:
            is_slave = request.service_is_readonly
        except AttributeError:
            is_slave = request.is_slave

        if is_slave is None:
            is_slave = slave.is_slave()
    else:
        is_slave = slave.is_slave()
    return {'is_slave': is_slave}


@register.filter
def html_quotation(s):
    if _lang() == 'ru':
        format_str = '«%s»'
    else:
        format_str = '“%s”'
    return format_str % str(s)


class SmartQuoteNode(template.Node):
    def __init__(self, nodelist):
        self.nodelist = nodelist
        super(SmartQuoteNode, self).__init__()

    def render(self, context):
        s = self.nodelist.render(context)
        return html_quotation(s)


@register.tag
def smartquote(parser, token):
    nodelist = parser.parse(['endsmartquote'])
    parser.next_token()
    return SmartQuoteNode(nodelist)


@register.simple_tag
def lquote():
    return '«' if _lang() == 'ru' else '“'


@register.simple_tag
def rquote():
    return '»' if _lang() == 'ru' else '”'


@register.simple_tag
def genderize(staff, male_variant, female_variant):
    if getattr(staff, 'gender', 'M') == 'F':
        return female_variant
    else:
        return male_variant


_months = translation.gettext('intranet_stuff.templatetags_RussianMonths').split(',')


@register.filter
def full_datetime(dt):
    if _lang() == 'ru':
        return '%d %s %d, %02d:%02d' % (dt.day, _months[dt.month - 1], dt.year, dt.hour, dt.minute)
    else:
        return dt.strftime('%H:%M, %B %d, %Y')


def _replace_page(url, page):
    """
    >>> _replace_url('http://host.com/list?page=1')
    'http://host.com/list'
    >>> _replace_url('http://host.com/list?page=1&order=title')
    'http://host.com/list?order=title'
    >>> _replace_url('http://host.com/list?section=qwerty&page=1&order=title')
    'http://host.com/list?section=qwerty&order=title'
    """
    if url.find(page) == -1:
        return url
    _url = url.replace(page, '')
    if _url[-1] in ('?', '&'):
        return _url[:-1]
    if _url.find('?&'):
        return _url.replace('?&', '?')
    return _url.replace('&&', '&')


_pager_tpl = """<lego:b-pager
    start-from="1"
    current-page="%d"
    total-count="%d"
    page-url="%s"
    page-param="%s"
    perpage-count="%s">
</lego:b-pager>
"""


@register.simple_tag(takes_context=True)
def lego_pager(context, paginator, page, var_name='page'):
    """
    http://wiki.yandex-team.ru/intranet/center/DjangoIntranetStuff/templatetags/intranetcommon#legopager
    """
    return _pager_tpl % (
        page.number,
        paginator.count,
        _replace_page(context.get('request').get_full_path(), '%s=%s' % (var_name, page.number)),
        var_name,
        paginator.per_page,
    )
