from pprint import pprint

import colorama

from django.db.models import Q
from django.db import transaction

from wiki.api_v2.exceptions import BadRequest
from wiki.async_operations.consts import OperationOwner, OperationIdentity
from wiki.async_operations.operation_executors.move_cluster.consts import (
    MoveCluster,
    MoveClusterLegacy,
    MoveClusterRequestWithLegacy,
)
from wiki.async_operations.operation_executors.move_cluster.move_cluster import (
    MoveClusterOperation,
    get_target_slugs,
    page_qs,
)
from wiki.pages.logic.backlinks import rewrite_links
from wiki.pages.logic.clone import clone_page
from wiki.pages.models import Page
from wiki.pages.models.cluster_change import AffectedPage
from wiki.pages.utils.remove import _mark_page_removed  # noqa
from wiki.sync.connect.org_ctx import org_ctx, _ORG_CTX_KEY, get_org, org_user  # noqa


colorama.init(autoreset=True)


class DummyProgressStorage:
    def report_begin_operation(self, *args, **kwargs):
        pass

    def report_failure(self, identity: OperationIdentity, result):
        print(f'FAILED: {result}')
        pass

    def report_progress(self, identity: OperationIdentity, f: float = 0.0, s: str = 'in progress'):
        print(f'Progress: {f}')

    def report_success(self, identity: OperationIdentity, result):
        print(f'SUCCESS: {result}')
        pass


def move_dry_run(src: str, dst: str, login: str, with_redirect=False, ignore_access_denied=False):
    m, d = _prepare_op(src, dst, login, with_redirect=with_redirect)
    return _move_dry_run(m, ignore_access_denied)


def move_execute(src: str, dst: str, login: str, with_redirect=False):
    m, d = _prepare_op(src, dst, login, with_redirect=with_redirect)
    m.execute(d)  # noqa


def _prepare_op(
    src: str,
    dst: str,
    login: str,
    with_redirect: bool = False,
) -> tuple[MoveClusterOperation, DummyProgressStorage]:
    user = org_user().get(username=login)

    if with_redirect:
        move_operation = MoveClusterLegacy(source=src, target=dst, with_children=True)
    else:
        move_operation = MoveCluster(source=src, target=dst)

    args = MoveClusterRequestWithLegacy(operations=[move_operation])
    owner = OperationOwner(user_id=user.id, org_inner_id=get_org() and get_org().id)

    m = MoveClusterOperation(args, owner)
    d = DummyProgressStorage()
    return m, d


def _move_dry_run(m: MoveClusterOperation, ignore_access_denied=False):
    try:
        m.check_preconditions()
    except BadRequest as e:
        print(colorama.Fore.RED + f'BAD REQUEST {e.error_code} {e.debug_message}')
        pprint(e.details)
        if e.error_code == 'CLUSTER_PERMISSION_DENIED' and ignore_access_denied:
            pass
        else:
            return False, e.details

    operation = m.args.operations[0]
    source_slugs_qs = page_qs(operation, org_id=m.owner.org_inner_id).values_list('supertag', flat=True)
    target_slugs = get_target_slugs(source_slugs_qs, source=operation.source, target=operation.target)

    print(colorama.Fore.GREEN + 'Good to go. New addresses:')
    pprint(target_slugs)

    return True, []


def rm_redirects(slugs):
    for slug in slugs:
        print(slug)
        page = Page.objects.get(supertag=slug)
        if page is None:
            continue
        if page.redirects_to is None:
            print('NOT A REDIRECT')
        if page.status == 0:
            print('ALREADY DELETED')

        _mark_page_removed(page)


def make_new_slug(slug: str, src: str, dst: str) -> str:
    return slug.replace(src, dst, 1)


def clone_cluster(source: str, target: str, user, dry_run=True, do_rewrite_links=True) -> list[Page]:
    active_pages = Page.objects.filter(org=get_org(), status__gt=0)

    query = Q(supertag=source) | Q(supertag__startswith=source + '/')
    pages = active_pages.filter(query).order_by('supertag')

    new_slugs = [make_new_slug(page.slug, src=source, dst=target) for page in pages]
    if intersection := active_pages.filter(supertag__in=new_slugs).values_list('supertag', flat=True):
        print(colorama.Fore.RED + 'Have intersection:\n' + '\n'.join(intersection))
        return []

    print(
        colorama.Fore.GREEN
        + f'Will clone [{pages.count()}] pages from [{source} -> {target}] by user [{user.username}]. '
        f'Rewrite links = [{do_rewrite_links}]'
    )

    if dry_run:
        for page in pages[:500]:
            print(f'{page.slug} -> {make_new_slug(page.slug, src=source, dst=target)}')

        print(colorama.Fore.GREEN + 'For start coping. Set parameter `dry_run=False`')
        return []

    print(colorama.Fore.GREEN + '\nStart coping')
    new_pages, affected = [], []
    with transaction.atomic():
        for page in pages:
            new_slug = make_new_slug(page.slug, src=source, dst=target)
            print(f'#{page.id} [{page.slug} -> {new_slug}]')

            new_page = clone_page(page, new_tag=new_slug, authors=[user], last_author=user, subscribe_me=False)
            affected.append(AffectedPage(id=new_page.id, slug=new_page.slug, previous_slug=page.slug))
            new_pages.append(new_page)

    print(colorama.Fore.GREEN + 'Pages copied')

    if do_rewrite_links:
        print(colorama.Fore.GREEN + 'Start rewrite links')
        rewrite_links(affected, org_id=get_org() and get_org().id)

    print(colorama.Fore.GREEN + 'Done')
    return new_pages
