# Сюда  положим хелперы для упрощения отладки наших приложений
import logging
from typing import List, Union, Optional

from django.contrib.auth import get_user_model  # noqa
from django.db.models import Q
from django.conf import settings

from wiki.api_frontend.serializers.grids.change_grid.change_grid import GridChangingUnit
from wiki.grids.models import Grid
from wiki.sync.connect import OPERATING_MODE_NAMES
from wiki.sync.connect.injector import get_from_thread_store, absorb, is_in_thread_store
from wiki.sync.connect.org_ctx import org_ctx, _ORG_CTX_KEY, get_org, org_user  # noqa
from wiki.sync.connect.tasks import ProcessChangeEventsTask
from wiki.sync.connect.models import Organization, OperatingMode
from wiki.pages.access import get_bulk_raw_access, interpret_raw_access
from wiki.pages.logic.move import move_clusters  # noqa
from wiki.pages.models import Access, Page, PageWatch  # noqa
from wiki.pages.utils.remove import delete_page
from wiki.pages.utils.resurrect import resurrect_page

from wiki.utils.backports.mds_compat import APIError
from wiki.favorites_v2.tasks import UpdateWatcherAutofolderTask
from wiki.utils.homepage_compressor import update_homepage  # noqa
from wiki.utils.subscription import generate_event
from wiki.utils.wiki_robot import get_wiki_robot
from wiki.ipython_move_toolkit import move_dry_run, move_execute, clone_cluster  # noqa


User = get_user_model()

logger = logging.getLogger(__name__)

if settings.IS_BUSINESS:
    from wiki.api_frontend.logic.forced_sync import forced_sync_now  # noqa
    from wiki.utils.provision_regress.model_serializer import provision_org  # noqa
    from wiki.sync.connect.dir_client import DirClient
    from wiki.sync.connect.tasks.helpers import import_dir_organization_mp_safe

    def import_org(dir_id: str):
        dir_client = DirClient()

        try:
            dir_org = dir_client.get_organization(dir_id, fields=settings.DIRSYNC_FIELDS_ORGANIZATION_FOR_SYNC)
            print(dir_org)
            if dir_org.get('organization_type') == 'portal':
                print('It is a portal! :x')
                return

            q = input('> I am running this console in Tmux or Screen (Y/N)')
            if q not in ('y', 'Y'):
                print('See ya')
                return

            import_dir_organization_mp_safe(dir_org, dir_id)
        except Exception:
            logger.exception(f'Can\'t import organization with dir_org_id={dir_id}')

    def use_org(dir_id: int):
        if not is_in_thread_store(_ORG_CTX_KEY):
            absorb(_ORG_CTX_KEY, [])

        org = Organization.objects.get(dir_id=dir_id)
        ctx = get_from_thread_store(_ORG_CTX_KEY)
        ctx.append(org)

        print('=' * 80)
        print(f'Organization: {org.name}, {org.label} dir_id={dir_id}')
        print('=' * 80 + '\n')

    def change_org_mode(dir_id: int, as_paid: bool = True):
        """Увеличить лимиты организации (страницы/файлы), сделав ее `платной`"""
        org = Organization.objects.get(dir_id=dir_id)

        mode_name = OPERATING_MODE_NAMES.paid if as_paid else OPERATING_MODE_NAMES.free
        mode = OperatingMode.objects.get(name=mode_name)
        org.mode = mode
        org.save()

        print(f'[{org.dir_id}] "{org.name}" change org.mode to [{mode_name}]')


def add_authors_to_cluster(*args, **kwargs):
    _change_authors_for_cluster('add', *args, **kwargs)


def remove_authors_from_cluster(*args, **kwargs):
    _change_authors_for_cluster('remove', *args, **kwargs)


def _change_authors_for_cluster(
    action: str,
    usernames: Union[str, List[str]],
    cluster: str,
    only_where_username: Optional[str] = None,
    org_id: Optional[int] = None,
    dry_run=True,
):
    if isinstance(usernames, str):
        usernames = [usernames]

    users = list(User.objects.filter(username__in=usernames))
    only_where = None

    if len(users) != len(usernames):
        raise ValueError(f'Not all usernames are found - {users}')

    if only_where_username:
        only_where = User.objects.get(username=only_where_username)

    cluster_args = (Q(supertag__startswith=cluster + '/') | Q(supertag=cluster)) & Q(org_id=org_id)

    if only_where:
        cluster_args = cluster_args & Q(authors=only_where)

    pages = list(Page.objects.filter(cluster_args))

    print('Following pages will be affected')
    for page in pages:
        print(f' - {page.supertag}')
    if not dry_run:
        print('Applying')
        for page in pages:
            if action == 'add':
                page.authors.add(*users)
            elif action == 'remove':
                page.authors.remove(*users)
            else:
                raise ValueError(f'Invalid action "{action}"')

        print('Done')


def resync_org(dir_id):
    q = ProcessChangeEventsTask()
    q.run(dir_id)


def get_next_revision(dir_id):
    from wiki.sync.connect.dir_client import DirClient
    from wiki.sync.connect.models import ChangeEvent

    dir_client = DirClient()
    revision = ChangeEvent.objects.get(org__dir_id=dir_id).revision
    print('Current revision = %s' % revision)
    params = {}
    if revision:
        params['revision__gt'] = revision
    events = dir_client.get_events(dir_id, **params)
    return events


def restore_page(supertag):
    page = Page.objects.get(supertag__endswith='/' + supertag, supertag__startswith='.deleted.', status=0)
    resurrect_page(page)


def fix_watchers(org):
    from wiki.pages.models import PageWatch

    usernames = list(set(User.objects.filter(orgs=org).values_list('username', flat=True)))
    PageWatch.objects.filter(page__org_id=org.id).exclude(user__in=usernames).values_list('user', flat=True)


def find_cluster(supertag):
    return list(Page.objects.filter(Q(supertag__startswith=supertag + '/') | Q(supertag=supertag)).all())


def delete_pages(pages):
    print(('Following %s pages will be removed:' % len(pages)))
    for page in pages:
        print((' - %s' % page.supertag))

    if input('\nPlease, enter YES to continue:') != 'YES':
        print('Cancelled by user')
        return

    for page in pages:
        delete_page(page)

    print('Purge complete')


def find_page(query):
    # supertag?
    try:
        return Page.objects.get(supertag=query, org=get_org())
    except Page.DoesNotExist:
        pass
    # id?
    try:
        return Page.objects.get(pk=int(query), org=get_org())
    except ValueError:
        pass
    except Page.DoesNotExist:
        pass


def _check_if_broken(page):
    try:
        page.body  # noqa
        return False
    except APIError:
        # missing indeed
        return True


def fix_bunch(supertags):
    for supertag in supertags:
        fix_ghost_grid(supertag)


def fix_ghost_grid(supertag, fix=True):
    page = find_page(supertag)

    if page is None:
        print('Can not find page')
        return
    print(('==== %s - mds %s' % (supertag, page.mds_storage_id.name)))

    if not _check_if_broken(page):
        print('Page looks fine')
        return

    for rev in page.revision_set.all().order_by('-created_at'):
        valid = not _check_if_broken(rev)
        name = 'Revision from %s by %s' % (rev.created_at, rev.author)

        if not valid:
            print('%s - current broken' % name)
            continue

        if fix:
            page.mds_storage_id.name = rev.mds_storage_id.name
            page.save(
                update_fields=[
                    'mds_storage_id',
                ]
            )

        print('%s Rolled back successfully' % (name))
        return

    print('[ no versions to rollback ]')


def explain_access(page):
    raw_access = get_bulk_raw_access([page])[page]
    return interpret_raw_access(raw_access)


def explain_membership(user):
    gr = set()
    first_level_groups_qs = user.staff.in_groups.filter(intranet_status=1)
    for group in first_level_groups_qs:
        loop = [group.name]
        gr.add(group)
        while group.parent is not None:
            group = group.parent
            loop.append(group.name)
            gr.add(group)
        print(' > '.join(loop))
    gr2 = set(user.get_all_groups())
    print((gr2 == gr))


def find_user(login):
    try:
        return org_user().get(username=login)
    except User.DoesNotExist:
        raise ValueError("Can't find user with a login %s" % login)


def find_user_everywhere(login):
    try:
        return get_user_model().objects.get(username=login)
    except User.DoesNotExist:
        raise ValueError("Can't find user with a login %s" % login)


def find_org(dir_id):
    return Organization.objects.get(dir_id=dir_id)


def unsubscribe_from_cluster(cluster: str, username: str):
    page = find_page(cluster)
    if page is None:
        print(f'Can not find page "{cluster}"')
        return

    user = find_user(username)

    queryset = PageWatch.objects.filter(
        Q(page__supertag=page.supertag, page__org=get_org())
        | Q(page__supertag__startswith=page.supertag + '/', page__org=get_org())
    )
    queryset = queryset.filter(user=user.username)
    queryset = queryset.only('page__supertag')
    supertags = [subscription.page.supertag for subscription in queryset]
    if len(supertags) > 0:
        queryset.delete()
        generate_event('unwatch', user, page, cluster=supertags)
        UpdateWatcherAutofolderTask().delay(user_id=user.id, username=user.username, org_id=get_org())

    print(f'User {username}@ was unsubscribed from {len(supertags)} pages in cluster "{cluster}"')


def remove_duplicate_column(page, column_id):
    robot_vika = get_wiki_robot()
    page = find_page(page)
    if page is None:
        print(f'Can not find page "{page}"')
        return

    grid = Grid.objects.get(id=page.id)
    operation = {'removed_column': {'name': column_id}}

    change_serializer = GridChangingUnit(grid, data=operation)
    change_serializer.save(robot_vika, grid, operation)
    # повторно удаляем дублирующися столбец
    change_serializer.save(robot_vika, grid, operation)


print('\nWiki Shell helpers loaded:')

print('- use_org(dir_id) !!!')
print('- Page, Access, User, Organization')
print('- explain_access')
print('- find_page (current ORG)')
print('- find_user (current ORG)')
print('- find_user_everywhere')
print('- find_org')
print('- find_cluster(supertag)')
print('- restore_page(supertag)')
print('- delete_pages([list_pages])')
print('- org_ctx')
print('- resync_org')
print('- move_clusters(user, clusters {old: new }, True)')
print('- add_authors_to_cluster(usernames, supertag, [only_where_username], [org_id], dry_run=True)')
print('- remove_authors_from_cluster(usernames, supertag, [only_where_username], [org_id], dry_run=True)')
print('- provision_org(user)')
print('- unsubscribe_from_cluster(cluster, user)')
print('- remove_duplicate_column(page, column_id)')
print('- update_homepage')
print('- move_dry_run, move_execute')
print('- change_org_mode(dir_id: int, as_paid: bool = True)')
print('- clone_cluster(source: str, target: str, user, dry_run=True, do_rewrite_links=True)')
print('- import_org(dir_id: str)')
