import tempfile
from base64 import encodebytes
from io import BytesIO, StringIO
from logging import getLogger
from os import path, remove
from subprocess import PIPE, Popen
from threading import Timer

from PIL import Image
from django.conf import settings

from wiki.legacy.diag_exceptions import ParseException
from wiki.legacy.formatters.base import BaseFormatter
from wiki.utils.errors import InputValidationError
from wiki.utils.tvm2 import get_service_ticket

logger = getLogger(__name__)

bin_dir = '/usr/bin'


class BaseGraphFormatter(BaseFormatter):
    KEY_SUPPORTED_FORMATS = '_supported_formats'
    supported_formats = ('html', 'svg')
    tools = []
    tpl_simple = '<img src="data:image/png;base64,%s" alt="" />'

    def _validate_params(self, params):
        content_format = params.get('output', 'html')
        _supported = params.get(self.KEY_SUPPORTED_FORMATS) or self.supported_formats
        if content_format not in _supported:
            raise InputValidationError('Unsupported "output" parameter value')

        width = params.get('width')
        height = params.get('height')
        if content_format == 'svg' and (width or height):
            raise InputValidationError('"output=svg" cannot be used with "width"/"height" parameters')

        try:
            width = int(width) if width else 0
            height = int(height) if height else 0
        except ValueError:
            raise InputValidationError('Invalid "width" or "height" parameter value')

        return content_format, width, height

    def format(self, text):
        """
        Функция возвращает или байты (для бинарных), или html... такие дела
        На это завязана логика внутри форматтера
        """
        content_format, width, height = self._validate_params(self.params)
        res, stderr = self.do_format(text)
        self.parse_stderr_from_tool(stderr)

        if width or height:
            if isinstance(res, bytes):
                stream = BytesIO(res)
            else:
                stream = StringIO(res)

            res = self.resize(stream, width, height)
            res = res.read()

        if content_format == 'html':
            if isinstance(res, str):
                res = res.encode()
            return self.tpl_simple % (encodebytes(res).decode('utf-8').replace('\n', ''))

        if content_format == 'svg':
            return res.decode()

        return res

    PANGO_WARNING_TEMPLATE = """'/root/.pangorc': Permission denied"""

    def strip_pango_warning(self, stderr):
        """
        Вернуть None, если библиотека Pango не писала ворнинг в stderr.
        Если библиотека все-таки писала, то вернуть strerr за вычетом этого ворнинга.
        Библиотека отвечает за рендеринг шрифтов.

        Пример:
        (process:18821): Pango-WARNING **: error opening config file '/root/.pangorc': Permission denied

        @type stderr: str
        @rtype: str
        """
        stderr = stderr.rstrip()
        found = stderr.rfind(self.PANGO_WARNING_TEMPLATE)
        if found == -1:
            return None
        found = stderr.rfind('(process')
        if found == -1:
            return None

        if found > 0:
            found -= 1  # потому что ошибки пишутся с переводом строки.

        return stderr[:found]

    def parse_stderr_from_tool(self, stderr):
        """
        Обработать строку, которая была напечатана в stderr.

        @type stderr: str
        @param stderr: результат работы консольной утилиты, который попал в stderr.
        @rtype: str
        @return: если None, то ошибки никакой нет
        @raise: InputValidationError
        """
        return

    def do_format(self, text):
        """
        Вернуть результат вызова форматтера графов.

        @type text: unicode
        @rtype: tuple
        """
        format = self.params.get('output')
        if format != 'svg':
            format = 'png'
        p = Popen([path.join(bin_dir, self.tool), '-T' + format], stdout=PIPE, stdin=PIPE, stderr=PIPE)
        p.stdin.write(text.encode('utf-8'))

        def kill_proc(p, timeout_holder):
            timeout_holder['value'] = True
            p.kill()

        timeout_holder = {'value': False}
        timer = Timer(30, kill_proc, [p, timeout_holder])  # 30 seconds
        timer.start()
        result, error = p.communicate()

        if isinstance(error, bytes):
            error = error.decode()

        timer.cancel()
        if timeout_holder['value']:
            logger.info(
                'Graph formatter timed out: tool "{tool}", format "{format}", text "{text}"'.format(
                    tool=self.tool, format=format, text=text
                )
            )
            raise Exception('"%s" tool process timed out' % self.tool)

        pango_stripped_out = self.strip_pango_warning(error)
        if pango_stripped_out:
            error = pango_stripped_out
        return result, error

    @property
    def tool(self):
        if len(self.ordered_params) and self.ordered_params[0] in self.tools:
            return self.ordered_params[0]
        return self.tools[0]

    def resize(self, buff, width, height):
        img = Image.open(buff).convert('RGB')
        src_width, src_height = img.size
        if not width or not height:
            if width:
                height = int((float(width) / src_width) * src_height)
            else:
                width = int((float(height) / src_height) * src_width)
        if width > src_width:
            width = src_width
        if height > src_height:
            height = src_height
        img = img.resize((width, height), Image.ANTIALIAS)
        res = BytesIO()
        img.save(res, 'PNG')
        res.seek(0)
        return res


class BaseDiagFormatter(BaseGraphFormatter):
    """
    Для blockdiag и seqdiag.
    (Наверное подойдет и для actdiag, nwdiag, rackdiag, packetdiag,
    но не изучал вопрос)
    """

    FONT_DIR = '/usr/share/fonts/truetype/'
    DEFAULT_FONT = 'msttcorefonts/arial.ttf'

    def get_tool_application_cls(self):
        """
        @return: parser, builder and drawer modules from diagram lib.
        """
        raise NotImplementedError()

    def do_format(self, text):
        format = self.params.get('output')
        if format != 'svg':
            format = 'png'

        font = self.params.get('font')
        if not font or not path.isfile(path.join(self.FONT_DIR, font)):
            font = self.DEFAULT_FONT

        _, diag_file_name = tempfile.mkstemp(prefix='wf_diagram_src', suffix='.diag')
        _, out_file_name = tempfile.mkstemp(prefix='wf_diagram_output', suffix='.' + format)
        with open(diag_file_name, 'wb') as file_obj:
            file_obj.write(text.encode('utf-8'))

        http_proxy = None
        if self.wf_config.use_http_proxy:
            user = settings.GOZORA_SOURCE
            tvm = get_service_ticket(settings.GOZORA_TVM2_CLIENT_ID)
            http_proxy = self.wf_config.http_proxy.format(user=user, tvm=tvm)

        try:
            app_cls = self.get_tool_application_cls()
            app_cls().run(
                [
                    '-T',
                    format,
                    '--antialias',
                    '--font',
                    path.join(self.FONT_DIR, font),
                    '-o',
                    out_file_name,
                    '--http-proxy',
                    http_proxy,
                    diag_file_name,
                ]
            )
        except ParseException:
            raise
        except Exception:
            raise
        else:
            with open(out_file_name, 'rb') as file_obj:
                res = file_obj.read()
            error = ''
        finally:
            if path.isfile(out_file_name):
                remove(out_file_name)
            if path.isfile(diag_file_name):
                remove(diag_file_name)

        return res, error
