

from datetime import datetime
from functools import wraps, partial
from json import dumps as real_dumps, loads

from django.db import models
from django.core.serializers.json import DjangoJSONEncoder
from django.utils.functional import Promise
from django.utils.encoding import force_str
from django.http import JsonResponse


class CustomJSONEncoder(DjangoJSONEncoder):
    def default(self, o):
        if isinstance(o, Promise):
            return force_str(o)
        elif isinstance(o, datetime):
            return o.strftime('%Y-%m-%dT%H:%M:%S')
        elif isinstance(o, models.Model):
            return force_str(o.pk)
        elif isinstance(o, set):
            return list(o)
        else:
            return super(CustomJSONEncoder, self).default(o)


dumps = partial(real_dumps, cls=CustomJSONEncoder)


def parse_datetime(str):
    return datetime.strptime(str, '%Y-%m-%dT%H:%M:%S')


def json_converter(attr_name, getter=None, setter=None):
    """
    Returns property object which wraps given attr_name with json load/dump
    """

    if getter is not None:

        def json_getter(instance):
            json_str = getattr(instance, attr_name)
            return getter(loads(json_str)) if json_str is not None else None

    else:

        def json_getter(instance):
            json_str = getattr(instance, attr_name)
            return loads(json_str) if json_str is not None else None

    if setter is not None:

        def json_setter(instance, dict_):
            setattr(instance, attr_name, setter(dumps(dict_)))

    else:

        def json_setter(instance, dict_):
            setattr(instance, attr_name, dumps(dict_))

    return property(json_getter, json_setter)


def returns_json(view):
    @wraps(view)
    def wrapper(*args, **kwargs):
        retval = view(*args, **kwargs)
        if isinstance(retval, dict):
            return JsonResponse(retval, encoder=CustomJSONEncoder)
        else:
            return retval

    return wrapper
