
from django.conf import settings


# Rules for degradation of languages.
langs_degradation = {
    'ru': 'ru',
    'en': 'en',
    'tr': 'en',
    'uk': 'ru',
    'kk': 'ru',
}

# All supported by current Django prjects languages.
supported_langs = [x[0] for x in getattr(settings, 'LANGUAGES', tuple())]


def get_default_lang():
    """
    Detect default language by project's settings.
    """
    return getattr(settings, 'LANGUAGE_CODE', 'en-us')[:2]


def get_supported_lang(lang, _supported_langs=None):
    """
    Checks if current language is supported and if isn't - degrade it.

    >>> get_supported_lang('ru', ['ru', 'en'])
    'ru'
    >>> get_supported_lang('kk')
    'ru'
    >>> get_supported_lang('en')
    'en'
    >>> get_supported_lang('aa')
    'en'
    """
    if _supported_langs is None:
        _supported_langs = supported_langs

    # Use current lang if one is supported.
    if lang in _supported_langs:
        return lang

    # Try to degrade current language and check if the result is supported.
    degraded_lang = langs_degradation.get(lang, get_default_lang())
    if degraded_lang in _supported_langs:
        return degraded_lang

    return get_default_lang()
