
import os
import logging

from django.core.files import locks
from django.conf import settings
from wiki.legacy.slave import is_slave
from django.core.management.base import BaseCommand


LOCK_PREFIX = getattr(settings, 'LOCK_PREFIX', None)
if not LOCK_PREFIX:
    LOCK_PATH = getattr(settings, 'LOCK_PATH', None)
    if not LOCK_PATH:
        LOCK_PATH = getattr(settings, 'VAR_PATH', None)
        if not LOCK_PATH:
            LOCK_PATH = '/tmp'

LOGGER_NAME = getattr(settings, 'PLOCKER_LOGGER_NAME', None)
if LOGGER_NAME:
    logger = logging.getLogger(LOGGER_NAME)
    LOGGER_LEVEL = getattr(settings, 'PLOCKER_LOGGER_LEVEL', logging.WARNING)


class PLocker(object):

    verbosity = 1

    def __init__(self):
        super(PLocker, self).__init__()

        self._filename = None
        self._filepointer = None

    def plock(self, cmd_id):
        if LOCK_PREFIX:
            self._filename = '/var/lock/' + LOCK_PREFIX + cmd_id.replace('/', '_')
        else:
            if not os.path.isdir(LOCK_PATH):
                os.makedirs(LOCK_PATH)
            self._filename = os.path.join(LOCK_PATH, cmd_id.replace('/', '_'))

        if os.path.isfile(self._filename):
            is_prev_unclean = True
        else:
            is_prev_unclean = False

        try:
            try:
                self._filepointer = open(self._filename, 'w')
            except IOError as e:
                if str(e).find('Permission denied') != -1:
                    self._log('Cannot create file "%s". Permission denied.' % self._filename)
                    return False
                else:
                    raise
            locks.lock(self._filepointer, locks.LOCK_EX | locks.LOCK_NB)
        except IOError:
            self._log('Another process detected: %s' % cmd_id)
            return False

        if is_prev_unclean:
            self._log('Previous process end unclean: %s' % cmd_id)

        return True

    def punlock(self):
        if self._filepointer and not self._filepointer.closed:
            locks.unlock(self._filepointer)
            self._filepointer.close()
        if self._filename and os.path.isfile(self._filename):
            os.remove(self._filename)

    def is_slave(self):
        return is_slave()

    def is_master(self):
        return not is_slave()

    def _log(self, msg):
        if LOGGER_NAME:
            logger.log(LOGGER_LEVEL, msg)
        elif self.verbosity:
            print(msg)


class PLockedCommand(PLocker, BaseCommand):

    plock_name = None
    check_is_slave = True

    def add_arguments(self, parser):
        parser.add_argument('--noplock', action='store_true', default=False, help='Do not use plocker.')

    def _get_default_lock_name(self):
        module_path = self.__class__.__module__
        if '.' in module_path:
            return module_path.rsplit('.', 1)[1]
        else:
            return module_path

    def handle(self, *args, **options):
        plock_name = self.plock_name if self.plock_name is not None else self._get_default_lock_name()

        if (
            options.get('noplock', False)
            or ((not self.check_is_slave) or (not self.is_slave()))
            and self.plock(plock_name)
        ):
            result = self.plocked_handle(*args, **options)
            self.punlock()
            return result

    def plocked_handle(self, *args, **options):
        raise NotImplementedError
