import re


class TranslitBase(object):

    translit_offset = ord('а')
    transtab = []
    upper_transtab = []
    yo_lower, yo_upper = 'yo', 'Yo'

    def translit(self, st):
        res = []
        tablen = len(self.transtab)
        if type(st) != str:
            st = str(st, 'utf-8')
        for c in st:
            islower = c.islower()
            clow = c.lower()
            tabpos = ord(clow) - self.translit_offset
            if 0 <= tabpos < tablen:
                res.append(self.transtab[tabpos] if islower else self.upper_transtab[tabpos])
            elif clow != 'ё':
                res.append(c)
            else:
                res.append(self.yo_lower if islower else self.yo_upper)
        return ''.join(res)


class Translit(TranslitBase):
    """
    Based on international standard ISO 9-95 (GOST 7.79-2000):

    http://www.gsnti-norms.ru/norms/common/doc.asp?0&/norms/stands/7_79.htm
    """

    transtab = [
        'a',
        'b',
        'v',
        'g',
        'd',
        'e',
        'zh',
        'z',
        'i',
        'j',
        'k',
        'l',
        'm',
        'n',
        'o',
        'p',
        'r',
        's',
        't',
        'u',
        'f',
        'x',
        'cz',
        'ch',
        'sh',
        'shh',
        '``',
        'y`',
        '`',
        'e`',
        'yu',
        'ya',
    ]

    upper_transtab = [
        'A',
        'B',
        'V',
        'G',
        'D',
        'E',
        'Zh',
        'Z',
        'I',
        'J',
        'K',
        'L',
        'M',
        'N',
        'O',
        'P',
        'R',
        'S',
        'T',
        'U',
        'F',
        'X',
        'Cz',
        'Ch',
        'Sh',
        'Shh',
        '``',
        'Y`',
        '`',
        'E`',
        'Yu',
        'Ya',
    ]

    czrepl = re.compile(r'(c|C)z([ieyjIEYJ])')

    def translit(self, st):
        st = super(Translit, self).translit(st)
        return self.czrepl.sub(r'\1\2', st)


translit = Translit().translit


class NameTranslit(TranslitBase):

    transtab = [
        'a',
        'b',
        'v',
        'g',
        'd',
        'e',
        'zh',
        'z',
        'i',
        'y',
        'k',
        'l',
        'm',
        'n',
        'o',
        'p',
        'r',
        's',
        't',
        'u',
        'f',
        'kh',
        'ts',
        'ch',
        'sh',
        'shch',
        '',
        'i',
        '',
        'e',
        'ju',
        'ya',
    ]
    upper_transtab = [
        'A',
        'B',
        'V',
        'G',
        'D',
        'E',
        'Zh',
        'Z',
        'I',
        'Y',
        'K',
        'L',
        'M',
        'N',
        'O',
        'P',
        'R',
        'S',
        'T',
        'U',
        'F',
        'Kh',
        'Ts',
        'Ch',
        'Sh',
        'Shch',
        '',
        'I',
        '',
        'E',
        'Ju',
        'Ya',
    ]


name_translit = NameTranslit().translit


class WikiTranslit(TranslitBase):

    macroses = {
        'вики': 'wiki',
        'вако': 'wacko',
        'вака': 'shwacko',
        'веб': 'web',
        'ланс': 'lance',
        'кукуц': 'kukutz',
        'мендокуси': 'mendokusee',
        'яремко': 'iaremko',
        'николай': 'nikolai',
        'алексей': 'aleksey',
        'анатолий': 'anatoly',
    }
    single_char_letters_from = [
        'а',
        'б',
        'в',
        'г',
        'д',
        'е',
        'з',
        'и',
        'к',
        'л',
        'м',
        'н',
        'о',
        'п',
        'р',
        'с',
        'т',
        'у',
        'ф',
        'х',
        'ъ',
        'ь',
        'ц',
        'ы',
        'А',
        'Б',
        'В',
        'Г',
        'Д',
        'Е',
        'З',
        'И',
        'К',
        'Л',
        'М',
        'Н',
        'О',
        'П',
        'Р',
        'С',
        'Т',
        'У',
        'Ф',
        'Х',
        'Ъ',
        'Ь',
        'Ц',
        'Ы',
    ]
    single_char_letters_to = [
        'a',
        'b',
        'v',
        'g',
        'd',
        'e',
        'z',
        'i',
        'k',
        'l',
        'm',
        'n',
        'o',
        'p',
        'r',
        's',
        't',
        'u',
        'f',
        'x',
        '',
        '',
        'c',
        'y',
        'A',
        'B',
        'V',
        'G',
        'D',
        'E',
        'Z',
        'I',
        'K',
        'L',
        'M',
        'N',
        'O',
        'P',
        'R',
        'S',
        'T',
        'U',
        'F',
        'X',
        '',
        '',
        'C',
        'Y',
    ]
    double_char_letters_from = [
        'й',
        'ё',
        'ж',
        'ч',
        'ш',
        'щ',
        'э',
        'ю',
        'я',
        'Й',
        'Ё',
        'Ж',
        'Ч',
        'Ш',
        'Щ',
        'Э',
        'Ю',
        'Я',
    ]
    double_char_letters_to = [
        'jj',
        'jo',
        'zh',
        'ch',
        'sh',
        'shh',
        'je',
        'ju',
        'ja',
        'Jj',
        'Jo',
        'Zh',
        'Ch',
        'Sh',
        'Shh',
        'Je',
        'Ju',
        'Ja',
    ]
    single_chars_length = len(single_char_letters_from)
    double_chars_length = len(double_char_letters_from)

    trash_re = re.compile(r'[^A-Za-z0-9\.\/\-]')

    def translit(self, word, to_lower=True):

        if type(word) == str:
            try:
                word = word.decode('utf-8')
            except UnicodeDecodeError:
                pass
        elif type(word) != str:
            return word

        # substitute macroses
        for macros_key in self.macroses:
            start = 0
            while True:
                position = word.lower().find(macros_key, start)
                start = position + 1
                if position > -1:
                    if word[position].lower() == word[position]:
                        continue
                    key_length = len(macros_key)
                    if (
                        len(word) > position + key_length
                        and word[position + key_length].lower() == word[position + key_length]
                    ):
                        continue
                    macros_value = self.macroses[macros_key]
                    word = word[:position] + macros_value.capitalize() + word[position + key_length :]
                else:
                    break

        # substitute single char translit chars
        for i in range(self.single_chars_length):
            word = word.replace(self.single_char_letters_from[i], self.single_char_letters_to[i])

        # substitute double char translit chars
        for i in range(self.double_chars_length):
            word = word.replace(self.double_char_letters_from[i], self.double_char_letters_to[i])

        # remove everything not allowed in the name
        r = self.trash_re.sub('', word).strip('/')
        return r.lower() if to_lower else r


wiki_translit = WikiTranslit().translit
