import logging
from typing import Union

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.auth.models import AnonymousUser
from django.http import HttpRequest, HttpResponse
from django_replicated.utils import routers
from django_yauth.middleware import YandexAuthMiddleware
from ylog.context import log_context

from wiki.api_core.errors.permissions import OrgOutOfSync, UserOutOfSync, DontCareAboutThisOrg
from wiki.api_core.errors.rest_api_error import BadRequest, RestApiError, UnauthorizedError, ForbiddenError
from wiki.api_core.utils import is_tvm_authentication, is_oauth2_authentication
from wiki.intranet.models import Staff
from wiki.middleware.org_detector import user_has_org
from wiki.middleware.wiki_auth import UserAuth
from wiki.sync.connect.models import Organization
from wiki.users import DEFAULT_AVATAR_ID

logger = logging.getLogger(__name__)

User = get_user_model()

FALSEY_VALUES = {'false', 'False', '0'}


class CloudAuthMiddleware(YandexAuthMiddleware):
    """
    Авторизирует облачного пользователя, от имени которого был сделан запрос в АПИ.
    """

    CONNECT_ROUTES = {
        '/_api/dir/changed',
    }
    NONAUTH_ROUTES = {
        '/unistat',
    }

    verbose = True

    client_can_request_as_anonymous = False
    client_can_request_as_noorg = False

    frontend_flow = False
    special_case_docviewer = False
    special_case_connect = False
    tvm_source = '?'

    def _verbose_log(self, msg):
        if not self.verbose:
            return
        logger.info(f'[CloudAuth] {msg}')

    def process_request(self, request):
        self.assign_yauser(request)

        with log_context(
            cloud_id=request.META.get('HTTP_X_CLOUD_UID'),
            uid=request.META.get('HTTP_X_UID'),
            org_id=request.META.get('HTTP_X_ORG_ID'),
            cloud_org_id=request.META.get('HTTP_X_CLOUD_ORG_ID'),
            tvm2_source=request.yauser.service_ticket.src
            if is_tvm_authentication(request) and request.yauser.is_authenticated()
            else '',
        ):
            try:
                if self.is_nonauth_route(request):
                    self.assign_user_no_auth(request)
                    self.assign_org_no_auth(request)
                    return

                if self.assign_oauth2(request):
                    pass
                else:
                    self.assign_tvm2_or_raise(request)

                self.assign_user(request)
                self.assign_org(request)
            except DontCareAboutThisOrg as e:
                return e.as_http_response()
            except RestApiError as e:
                logger.warning(f'{e.status_code}: {e.debug_message}')
                return e.as_http_response()

    def is_nonauth_route(self, request: HttpRequest):
        path = request.path

        if path in self.CONNECT_ROUTES:
            self.special_case_connect = True
            return True

        if path in self.NONAUTH_ROUTES:
            return True

        return False

    def assign_oauth2(self, request: HttpRequest):
        if not is_oauth2_authentication(request):
            return False

        if not request.yauser.is_authenticated():
            # скорее всего пользователь корректный, но мы про него не знаем в Вики.
            # На Forced Sync направлять смысла нет, пусть зайдут через фронт
            raise OrgOutOfSync(debug_message='OAuth was unsuccessful; Please authenticate user via frontend first')

        return True

    def assign_tvm2_or_raise(self, request: HttpRequest):
        if not is_tvm_authentication(request):
            raise UnauthorizedError(
                debug_message='TVM2 authentication required;' + f'[{request.method}] - {request.get_full_path()}'
            )

        if not request.yauser.is_authenticated():
            # запрос не прошел аутентификацию по tvm2
            raise ForbiddenError(debug_message='TVM2 service ticket is invalid')

        tvm_source = request.yauser.service_ticket.src

        if tvm_source not in settings.YAUTH_TVM2_ALLOWED_CLIENT_IDS:
            # доверяем только определенному перечню tvm клиентов
            raise ForbiddenError(debug_message=f'Not allowed tvm client: {tvm_source}')

        self.client_can_request_as_anonymous = tvm_source in settings.TVM2_CLIENTS_WITHOUT_USER_TICKET
        self.client_can_request_as_noorg = tvm_source in settings.TVM2_CLIENTS_WITHOUT_ORG_ID

        self.frontend_flow = str(tvm_source) == str(settings.WIKI_FRONT_TVM2_CLIENT_ID)
        self.special_case_docviewer = str(tvm_source) == str(settings.DOCVIEWER_TVM2_CLIENT_ID)
        self.tvm_source = tvm_source

    def assign_user_no_auth(self, request: HttpRequest) -> Union[HttpResponse, None]:
        request.user = AnonymousUser()
        request.user.avatar_id = DEFAULT_AVATAR_ID
        request.user_auth = UserAuth.from_request(request)

    def assign_user(self, request: HttpRequest):
        request.user = AnonymousUser()
        request.user.avatar_id = DEFAULT_AVATAR_ID

        uid = request.META.get('HTTP_X_UID')
        cloud_uid = request.META.get('HTTP_X_CLOUD_UID')

        if request.yauser and request.yauser.is_authenticated():
            uid_from_userticket = request.yauser.uid
            if uid_from_userticket and not uid:
                self._verbose_log(f'Got user ticket for user {uid_from_userticket}; will be use instead of header')
                uid = uid_from_userticket

        if not uid and not cloud_uid:
            if self.client_can_request_as_anonymous:
                self._verbose_log('Requested as anonymous user')
                request.user_auth = UserAuth.from_request(request)
                return
            raise BadRequest(debug_message='"X-CLOUD-UID" or "X-UID" header is required')

        request.user = get_user_by_cloud_uid(cloud_uid) or get_user_by_uid(uid, cloud_uid) or request.user

        if not self.has_authenticated_wiki_user(request):
            raise UserOutOfSync(debug_message=f'No user found for UID:{uid} CloudUID:{cloud_uid}')

        request.user_auth = UserAuth.from_request(request)

    def _get_org_id(self, request: HttpRequest) -> str:
        org_id = request.META.get('HTTP_X_ORG_ID')

        if self.special_case_docviewer:
            # Если к нам пришел DocViewer, то у него org_id не в заголовке, а зашит в параметре fileid:
            # fileid=page_supertag/file_url?org_id
            fileid = request.GET.get('fileid', '')
            logger.info(f'Getting ORGID for Docviewer from ${fileid}')
            fileid_tokens = fileid.split('?')
            if len(fileid_tokens) == 2:
                org_id = fileid_tokens[1]

        return org_id

    def _get_cloud_org_id(self, request: HttpRequest) -> str:
        cloud_org_id = request.META.get('HTTP_X_CLOUD_ORG_ID')

        # todo special cases

        return cloud_org_id

    def do_assign_org(self, request: HttpRequest, org_id=None, cloud_org_id=None):
        if org_id:
            request.org = Organization.objects.get(dir_id=org_id)
        else:
            request.org = Organization.objects.get(cloud_id=cloud_org_id)

    def assign_org_no_auth(self, request: HttpRequest) -> Union[HttpResponse, None]:
        # обходной маневр для ручек без tvm2 авторизации (директория, юнистат)
        request.org = None
        org_id = self._get_org_id(request)
        cloud_org_id = self._get_cloud_org_id(request)

        if org_id or cloud_org_id:
            try:
                self.do_assign_org(request, org_id, cloud_org_id)
            except Organization.DoesNotExist:
                if self.special_case_connect:
                    raise DontCareAboutThisOrg(
                        debug_message=f'Organization(id={org_id}, cloud_id={cloud_org_id}) does not exist; Ignoring update'
                    )
                else:
                    raise OrgOutOfSync(
                        debug_message=f'Organization(id={org_id}, cloud_id={cloud_org_id}) does not exist'
                    )

    def assign_org(self, request: HttpRequest) -> Union[HttpResponse, None]:
        request.org = None
        org_id = self._get_org_id(request)
        cloud_org_id = self._get_cloud_org_id(request)

        # Нам не передали хедер
        if not (org_id or cloud_org_id):
            if self.frontend_flow and request.user.is_authenticated:
                raise UserOutOfSync(debug_message=f'Frontend claims that {request.user} has no orgs')
            if self.client_can_request_as_noorg:
                self._verbose_log('Requested as a noorg')
                return

            raise BadRequest(debug_message='"X-Org-Id" header value is required')

        # Пробуем найти организацию с переданным id в базе Вики
        try:
            self.do_assign_org(request, org_id, cloud_org_id)
        except Organization.DoesNotExist:
            raise OrgOutOfSync(debug_message=f'Organization(id={org_id},cloud_id{cloud_org_id}) does not exist')

        if request.org.status != Organization.ORG_STATUSES.enabled:
            raise OrgOutOfSync(debug_message=f'Organization with org_id={org_id} is disabled')

        if request.user.is_authenticated and not user_has_org(request.user, request.org.dir_id):
            raise UserOutOfSync(
                debug_message=f'User {request.user} not found in organization dir_id={request.org.dir_id}'
            )

    @classmethod
    def has_authenticated_wiki_user(cls, request: HttpRequest) -> bool:
        return hasattr(request, 'user') and request.user and request.user.is_authenticated


def get_user_by_uid(uid: str, cloud_uid: str) -> Union[Staff, None]:
    try:
        staff = Staff.objects.select_related('user').get(uid=uid, is_dismissed=False)
        user = staff.user
        update_cloud_uid(user, cloud_uid)
        return user
    except Staff.DoesNotExist:
        return None


def update_cloud_uid(user: User, cloud_uid: str):
    if cloud_uid and not user.cloud_uid:
        # Обновим cloud_uid у пользователя данными из запроса
        try:
            routers.use_state('master')
            User.objects.filter(id=user.id).update(cloud_uid=cloud_uid)  # сохранение пользователя обвешано сигналами!
        finally:
            routers.revert()


def get_user_by_cloud_uid(cloud_uid: str) -> Union[User, None]:
    if not cloud_uid:
        return None

    try:
        return User.find_by_cloud_uid(cloud_uid)
    except User.DoesNotExist:
        return None
