import logging

from django.conf import settings
from django.http import HttpResponseBadRequest, HttpResponseForbidden
from django.utils.deprecation import MiddlewareMixin

from wiki.sync.connect.models import Organization
from wiki.api_core.utils import is_tvm_authentication
from wiki.api_core.errors.permissions import UserDoesNotHaveWiki
from wiki.middleware.wiki_auth import user_has_org
from wiki.org import get_user_orgs
from wiki.utils.employees import get_user_orgs_from_blackbox


logger = logging.getLogger(__name__)


# TODO Удалить после переезда в облако
class OrgDetectorMiddleware(MiddlewareMixin):
    """
    Middleware, определяющая, для какой организации сделан запрос.
    """

    def process_request(self, request):
        request.org = None

        if not settings.IS_BUSINESS:
            # в интранете нет организаций
            return

        if not is_tvm_authentication(request):
            # в B2B с контекстом организации пускаем только с tvm2 тикетами
            return

        if is_tvm_authentication(request) and not request.yauser.is_authenticated():
            # запрос не прошел аутентификацию по tvm2
            return HttpResponseForbidden(content='TVM2 service ticket is invalid')

        if (
            not request.user_auth.tvm2_user_ticket
            and request.yauser.service_ticket.src not in settings.TVM2_CLIENTS_WITHOUT_USER_TICKET
            and not request.user.cloud_uid
        ):
            # без tvm2 user тикета позволяем приходить только некоторым сервисам
            # и федеративным пользователям
            return HttpResponseForbidden(content='TVM2 user ticket is required')

        dir_org_id = self.get_dir_org_id(request)

        # Нам не передали хедер, или эвристика не сработала
        if not dir_org_id:
            # Мы разрешаем некоторым сервисам ходить без контекста организации
            if request.yauser.service_ticket.src in settings.TVM2_CLIENTS_WITHOUT_ORG_ID:
                return
            else:
                return HttpResponseBadRequest(content='"X-Org-Id" header value is required')

        # Нам передали хедер, попробуем ракрутить организацию
        try:
            request.org = Organization.objects.get(dir_id=dir_org_id)
            if request.org.status != Organization.ORG_STATUSES.enabled:
                # если у организации не подключен сервис Вики или она удалена, то сразу редиректим на Коннект
                return UserDoesNotHaveWiki(reason='deactivated').as_http_response()

            if (
                request.user.is_anonymous
                and request.yauser.service_ticket.src in settings.TVM2_CLIENTS_WITHOUT_ORG_ID
            ):
                # поиск зачем-то приходит в ручку .orgs-with-changes c X-Org-Id
                return

            if request.user.is_anonymous or not user_has_org(request.user, request.org.dir_id):
                # в организации нет такого пользователя
                # user.is_anonymous == True - пользователь не найдет в базе Вики
                return UserDoesNotHaveWiki(reason='not-imported-user').as_http_response()

        except Organization.DoesNotExist:
            logger.warning(
                'Organization with dir_org_id=%s does not exist. Request path=%s', dir_org_id, request.get_full_path()
            )

            if str(request.yauser.service_ticket.src) == settings.WIKI_FRONT_TVM2_CLIENT_ID:
                # пришел запрос из Вики фронтенда
                if dir_org_id in get_user_orgs_from_blackbox(request.yauser):
                    # такая организация есть у пользователя в ЧЯ, а у нас еще нет
                    return UserDoesNotHaveWiki(reason='not-imported-org').as_http_response()
                else:
                    # такой организации нет ни у нас, ни в ЧЯ
                    return UserDoesNotHaveWiki(reason='external-user').as_http_response()
            else:
                return HttpResponseBadRequest(
                    content='"X-Org-Id" header value is incorrect: organization with id="%s" not found' % dir_org_id
                )

    @classmethod
    def get_dir_org_id(cls, request):
        dir_org_id = request.META.get('HTTP_X_ORG_ID', '')
        if str(request.yauser.service_ticket.src) == settings.DOCVIEWER_TVM2_CLIENT_ID:
            # Если к нам пришел DocViewer, то у него org_id не в заголовке, а зашит в параметре fileid:
            # fileid=page_supertag/file_url?org_id
            fileid_tokens = request.GET.get('fileid', '').split('?')
            if len(fileid_tokens) == 2:
                dir_org_id = fileid_tokens[1]
            else:
                # TODO костыль для решения проблемы WIKI-14251
                # После решения задачи WIKI-14319 (добавления org_id в docviewer_url на фронте Вики)
                # этот кейс надо удалить
                dir_org_id = get_user_orgs(request.user)[0].dir_id

        return dir_org_id
