import logging

from django.conf import settings
from django.contrib.auth.models import AnonymousUser
from django_yauth.middleware import YandexAuthMiddleware
from wiki.api_core.utils import is_tvm_authentication
from wiki.intranet.models import Staff
from wiki.middleware.wiki_auth import UserAuth

logger = logging.getLogger(__name__)


class PassportAuthMiddleware(YandexAuthMiddleware):
    """
    Аутентификация в паспорте через django_yauth, получение дополнительных полей из паспорта. Не ходит в blackbox,
    если пользователь уже был авторизован другой middleware.
    """

    def process_request(self, request):
        if hasattr(request, 'user') and request.user.is_authenticated:
            logger.debug('PassportAuthMiddleware user is authenticated')
            return

        # Тут может выскочить ошибка аутентификации, например, TwoCookiesRequired, и ее никто не перехватит
        try:
            # YandexAuthMiddleware - класс старого стиля, super() не работает
            response = YandexAuthMiddleware.process_request(self, request)

            if request.yauser.is_authenticated():
                # при авторизации по TVM2 тикетам в fields ничего нет, так как BB не ходим, поэтому язык пользователя
                # достанем из нашей базы позже в LocaleMiddleware
                # TODO после полного перехода на TVM2 инициализацию языка здесь можно будет убрать совсем
                request.LANGUAGE_CODE = request.yauser.fields.get('language')
        except Exception as exception:
            logger.debug('PassportAuthMiddleware exception')
            response = self.process_exception(request, exception)
            if response:
                logger.debug('PassportAuthMiddleware exception processed')
                return response

            raise

        # атрибут для удобного доступа к аутентификационным данным, чтобы
        # передавать его в функции, где эти данные нужны, но не хочется
        # передавать request целиком
        request.user_auth = UserAuth.from_request(request)

        if request.user:
            request.user.avatar_id = request.user_auth.avatar_id

        logger.debug('PassportAuthMiddleware out')
        return response

    def assign_user(self, request):
        request.user = self._get_user(request)

    def _get_user(self, request):
        user = None

        if request.yauser.is_authenticated():
            uid = request.yauser.uid

            filter_params = {}
            if uid:
                filter_params['uid'] = uid
            elif is_tvm_authentication(request):
                uid = request.GET.get('uid')
                cloud_id = request.META.get('HTTP_X_CLOUD_UID')  # Поддерживаем федеративных пользователей

                if uid:
                    filter_params['uid'] = uid
                elif cloud_id:
                    filter_params['user__cloud_uid'] = cloud_id

            if not filter_params:
                return AnonymousUser()

            try:
                if 'is_dismissed' in getattr(settings, 'INTRANET_MODELS_STAFF', []):
                    filter_params['is_dismissed'] = False
                staff = Staff.objects.select_related('user').get(**filter_params)
            except Staff.DoesNotExist:
                logger.warning('Staff with uid %s was not found', uid)
                user = None
            else:
                user = staff.user
                if user is None:
                    # someone is wrong in the database
                    logging.warning('Staff with uid %s not linked with User', uid)
                else:
                    # благодаря select_related request.user.staff и request.user.staff.user
                    # не будут делать дополнительных запросов (с django 1.6 проставляется кэш)
                    # так как у нас еще местами дергается request.user.staff,
                    # заполним этот кэш для избежания запросов
                    user._profile_cache = staff

        return user or AnonymousUser()
