from django.conf import settings
from django.contrib.auth.models import AnonymousUser
from django.utils.deprecation import MiddlewareMixin
from django_replicated.utils import routers
from wiki.api_core.errors.rest_api_error import BadRequest

from wiki.org import org_ctx
from wiki.personalisation.user_cluster import create_personal_page, has_user_personal_cluster


class PersonalClusterInspectorMiddleware(MiddlewareMixin):
    """
    Middleware, создающая пользовательский кластер, если его еще нет.
    """

    def process_request(self, request):
        user = request.user
        if isinstance(user, AnonymousUser):
            return

        if settings.IS_BUSINESS:
            # в B2B проверяем явно наличие пользовательского кластера, так как значение флага has_personal_cluster
            # актуально только для одной организации, а пользователь в B2B может состоять в нескольких организациях
            if request.org is None:  # ---- иначе будет 500 NOORG
                return BadRequest(debug_message='"X-Org-Id" header value is required').as_http_response()

            with org_ctx(request.org):
                if not user.is_dir_robot and not has_user_personal_cluster(user):
                    self.create_personal_cluster(user)
        elif not user.has_personal_cluster:
            with org_ctx(request.org):
                self.create_personal_cluster(user)

    @staticmethod
    def create_personal_cluster(user):
        routers.init('master')

        create_personal_page(user)

        user.has_personal_cluster = True
        user.save()
