import logging

from django.conf import settings
from django.core.cache import caches
from django.db import DEFAULT_DB_ALIAS
from django.utils.deprecation import MiddlewareMixin
from django_replicated.dbchecker import db_is_alive

from wiki.utils.per_request_memoizer import memoize_for_this_request

logger = logging.getLogger('wiki_read_only')
cache = caches['wiki_state']


class ReadOnlyFlagMiddleware(MiddlewareMixin):
    """
    При отвалившемся мастере выставляем флаг read_only и переводим
    все запросы на слейв
    """

    def process_request(self, request):
        request.service_is_readonly = service_is_readonly()

        # для обратной совместимости
        request.is_slave = request.service_is_readonly


@memoize_for_this_request
def service_is_readonly(consistent_answer=False):
    """
    Return True if master is unavailable

    @type consistent_answer: bool
    @rtype: bool
    """
    allowed_downtime = getattr(settings, 'DATABASE_DOWNTIME', 60)  # cache for db_is_alive cmd
    # потому что используется класс TwoLevelCache
    manual_state = cache.get('read_only', False, second_level_only=consistent_answer)

    return manual_state or not db_is_alive(DEFAULT_DB_ALIAS, cache_seconds=allowed_downtime, number_of_tries=3)
