from django.utils.deprecation import MiddlewareMixin
from django_replicated.middleware import ReplicationMiddleware as OriginalReplicationMiddleware
from django_replicated.utils import routers

from wiki.api_core.utils import parse_url
from wiki.arc_compat import is_arc

SLAVE_HTTP_METHODS = ('GET', 'HEAD')
STATE_MASTER = 'master'
STATE_SLAVE = 'slave'

if is_arc():

    class ReplicationMiddlewareBase(OriginalReplicationMiddleware):
        pass


else:

    class ReplicationMiddlewareBase(MiddlewareMixin, OriginalReplicationMiddleware):
        pass


class ReplicationMiddleware(ReplicationMiddlewareBase):
    def process_request(self, request):
        routers.init(self.select_state(request))

    def select_state(self, request):
        """
        Выбрать состояние из пары 'master' и 'slave'.

        @param request:
        @rtype: str
        """
        state = STATE_SLAVE if request.method in SLAVE_HTTP_METHODS else STATE_MASTER
        state = self.check_state_override(request, state)
        state = manually_override_state(request, state)
        return state

    def process_response(self, request, response):
        routers.revert()
        return response


MASTER_ONLY_HANDLERS = ('watch', 'unwatch', 'masswatch', 'massunwatch', 'reformat', 'import', 'importpreview', 'edit')

SLAVE_ONLY_HANDLERS = ('explainlinks',)


def manually_override_state(request, state):
    """
    Переопределить выбор состояния в зависимости от логики работы вики.

    @type request: HttpRequest
    @type state: str
    @rtype: str
    @todo: Заменить на REPLICATED_VIEWS_OVERRIDES когда избавимся от dispatcher.dispatch.
    """
    parsed_url = parse_url(request.META['PATH_INFO'])
    handler_name = parsed_url.get('handler_name').lower()

    if request.method == 'POST' and state == STATE_MASTER:
        # explainlinks не пишет, только читает. Поэтому ходит в slave.
        if handler_name in SLAVE_ONLY_HANDLERS:
            state = STATE_SLAVE

    elif request.method in SLAVE_HTTP_METHODS and state == STATE_SLAVE:
        # Хендлеры Вики, для которых всегда используется master.
        if handler_name in MASTER_ONLY_HANDLERS:
            state = STATE_MASTER

    if request.method == 'GET' and '_for_update' in request.GET:
        state = STATE_MASTER

    return state
