
import re
import logging
from dataclasses import dataclass
from typing import Optional, Union

from django.contrib.auth.models import AnonymousUser
from django.contrib.auth import get_user_model

from django_yauth import util as yauth_util

from wiki.sync.connect.models import Organization
from wiki.api_core.utils import is_tvm_authentication
from wiki.users import DEFAULT_AVATAR_ID
from wiki.users.abstract_user import AbstractWikiUser

logger = logging.getLogger(__name__)

User = get_user_model()


@dataclass
class UserAuth(object):
    user: Union[AnonymousUser, AbstractWikiUser]
    user_ip: str
    server_host: str

    uid: Optional[str] = None
    sessionid: Optional[str] = None
    oauth_token: Optional[str] = None
    tvm_ticket: Optional[str] = None
    tvm2_user_ticket: Optional[str] = None
    cloud_uid: Optional[str] = None
    avatar_id: str = DEFAULT_AVATAR_ID
    iam_token: Optional[str] = None
    is_tvm_authentication: bool = False

    @classmethod
    def from_request(cls, request):
        # проверяем атрибут _yauser вместо yauser, потому что дескриптор yauser
        # ставится на класс и если он существует нельзя сделать вывод
        # обработался ли запрос YandexAuthMiddleware, атрибут _yauth же ставится
        # на инстанс request, когда запрос действительно обработался через
        # YandexAuthMiddleware и было обращение к yauth.

        yauser = getattr(request, '_yauser', None)
        is_tvm = False

        if yauser:
            # old backend
            uid = yauser.uid
            tvm_ticket = yauser.ticket if yauser.is_authenticated() else None
            tvm2_user_ticket = yauser.raw_user_ticket if yauser.is_authenticated() else None
            is_tvm = is_tvm_authentication(request)
            avatar_id = yauser.fields.get('default_avatar_id') if yauser.is_authenticated() else DEFAULT_AVATAR_ID
            iam_token = cls.get_iam_token(request)
        else:
            # new api backend
            # @todo это какая-то дичь, надо залоггировать кто ее использует (tvm/oauth) + выпилить
            uid = request.GET.get('__uid', '')
            tvm_ticket = None
            tvm2_user_ticket = None
            avatar_id = DEFAULT_AVATAR_ID
            iam_token = None

        oauth_token = None
        if 'HTTP_AUTHORIZATION' in request.META and request.META['HTTP_AUTHORIZATION'].startswith('OAuth '):
            oauth_token = request.META['HTTP_AUTHORIZATION'][len('OAuth '):]

        if not uid and request.user.is_authenticated:
            try:
                uid = request.user.get_uid()
            except Exception as e:
                logger.exception(f'get UID failed: {e}')

        return cls(
            user=request.user,
            uid=uid,
            cloud_uid=request.META.get('HTTP_X_CLOUD_UID', None),
            sessionid=UserAuth.get_session_id_from_cookies(request.COOKIES),
            oauth_token=oauth_token,
            tvm_ticket=tvm_ticket,
            tvm2_user_ticket=tvm2_user_ticket,
            user_ip=yauth_util.get_real_ip(request),
            server_host=yauth_util.get_current_host(request),
            avatar_id=avatar_id,
            iam_token=iam_token,
            is_tvm_authentication=is_tvm,
        )

    @classmethod
    def get_session_id_from_cookies(cls, cookies):
        sessionid_name = yauth_util.get_setting(['SESSIONID_COOKIE_NAME', 'YAUTH_SESSIONID_COOKIE_NAME'])
        return cookies.get(sessionid_name)

    @classmethod
    def get_iam_token(cls, request):
        auth = request.META.get('HTTP_AUTHORIZATION')
        if auth:
            iam_auth_token_regex = r'Bearer (?P<token>.+)'
            match = re.match(iam_auth_token_regex, auth)
            if match:
                return match.group('token')

        return None

    def __repr__(self):
        return 'User user: "{0}", uid: "{1}", cut session id: "{2}"'.format(
            self.user, self.uid, self.sessionid if not self.sessionid else self.sessionid[:3]
        )

    def get_oauth_token(self):
        if self.oauth_token:
            return self.oauth_token

        from wiki.utils import oauth

        self.oauth_token = oauth.get_token_by_user_auth(user_auth=self)
        return self.oauth_token


def user_has_org(user: User, dir_org_id: str) -> bool:
    return user.orgs.filter(status=Organization.ORG_STATUSES.enabled, dir_id=dir_org_id).exists()
