import logging

import requests
from requests.adapters import HTTPAdapter
from requests.exceptions import RequestException, Timeout

from django.conf import settings


logger = logging.getLogger(__name__)

MAX_REQ_ATTEMPTS = 3


class BunkerClient:
    def __init__(self, max_retries=3):
        self.session = requests.Session()
        self.session.mount('http://', HTTPAdapter(max_retries))

    def _get_response_data(self, url: str):
        attempt_count = 0
        while attempt_count < MAX_REQ_ATTEMPTS:
            attempt_count += 1
            try:
                res = self.session.get(url)
                if res.status_code == 200:
                    return res.json()
                else:
                    logger.error('Error in bunker api response: {}, url {}'.format(res, url))
                    break
            except (RequestException, Timeout) as ex:
                logger.error(f'Some problem with request to bunker api: {str(ex)}, let\'s try again')
                continue
            except Exception:
                logger.exception('Got error while requesting bunker api.')
                break

        return None

    def get_node_content(self, xpath: str, version='stable') -> dict:
        url = f'http://{settings.BUNKER_HOST}/v1/cat/?node={xpath}&version={version}'
        return self._get_response_data(url)

    def get_node_children(self, xpath: str, recursively=True, version='stable') -> dict:
        endpoint = 'tree' if recursively else 'ls'
        url = f'http://{settings.BUNKER_HOST}/v1/{endpoint}/?node={xpath}&version={version}'
        return self._get_response_data(url)
