import logging
import re
from typing import Dict

from django.core.cache import caches

from wiki.morda.bunker import BunkerClient
from wiki.morda.consts import BunkerNodeVersions

logger = logging.getLogger(__name__)

URL_IN_MIME_TYPE_REGEX = re.compile(r'(.*)avatar-href=\"(?P<url>.*)\";(.*)', re.DOTALL)
WIDGETS_TIMEOUT = 10 * 60  # in minutes
CACHE_WIDGETS_DATA_BACKEND = caches['widgets_data']
BUNKER_PROJECT = 'wiki-morda-intraneta'


def _is_deleted_node(node: Dict[str, bool]) -> bool:
    return node.get('isDeleted')


def _get_img_url_from_mime_type(mime):
    match = re.match(URL_IN_MIME_TYPE_REGEX, mime or '')
    if match:
        return match.groupdict()['url']
    else:
        return ''


def get_widget_data(xpath, ignore_cache=False) -> Dict[str, dict]:
    full_name = f'/{BUNKER_PROJECT}/{xpath}'
    version = BunkerNodeVersions.LATEST if ignore_cache else BunkerNodeVersions.STABLE  # TODO latest for testing ?
    tree = get_tree(ignore_cache, version)
    data = tree['content'].get(full_name) or {}
    images = {
        key.split('/')[-1]: tree['images'][key] for key in tree['images'].keys() if key.startswith(f'{full_name}/')
    }
    result = {'content': data, 'images': images}
    return result


def _get_tree_from_bunker(version) -> Dict[str, dict]:
    data = {}
    images = {}
    client = BunkerClient()
    node_list = client.get_node_children(f'/{BUNKER_PROJECT}', version=version)
    for node in node_list:
        if not _is_deleted_node(node) and 'mime' in node:
            mime = node['mime']
            full_name = node['fullName']
            if mime.startswith('application/json;'):
                content = client.get_node_content(full_name, version=version)
                data[full_name] = content
            elif mime.startswith('image/'):
                images[full_name] = _get_img_url_from_mime_type(mime)

    return {'content': data, 'images': images}


def get_tree(ignore_cache: bool, version: BunkerNodeVersions):
    tree = CACHE_WIDGETS_DATA_BACKEND.get('widgets_data')
    if tree and not ignore_cache:
        return tree

    try:
        tree = _get_tree_from_bunker(version)
        if tree:
            CACHE_WIDGETS_DATA_BACKEND.set('widgets_data', tree, WIDGETS_TIMEOUT)
            # TODO save in postgresql
            return tree
    except Exception as exc:
        logger.exception(repr(exc))

    # TODO get tree from postgresql
