import json
import logging

from rest_framework.response import Response

from wiki.api_core.raises import raises
from wiki.morda.consts import CONTENT_WIDGET_ORDERING, WidgetTypes
from wiki.morda.errors import JsonValidationError
from wiki.morda.datasources import bunker_datasource
from wiki.morda.widgets.registry import WIDGET_REGISTRY
from wiki.morda.views.base import MainPageAPIView
from wiki.morda.serializers import WidgetTypesListSerializer


logger = logging.getLogger(__name__)


class MainPageContentView(MainPageAPIView):
    serializer_class = WidgetTypesListSerializer

    @raises(JsonValidationError)
    def get(self, request, *args, **kwargs):
        """
        Получить контент всей страницы (кроме персональных виджетов и фидов из этушки).
        Запрашиваемый контент можно фильтровать по списку типов виджетов с помощью параметра widget_types.
        Если параметр widget_types отсутствует, то ручку отдает контент всех доступных виджетов.
        По-умолчанию для выгрузки данных используется кэширование данных из бункера на стороне бекенда.
        Для выгрузки данных напрямую из бункера необходимо передать флаг ignore_cache=True.

        %%
        curl -H "Authorization: OAuth <token>" "https://wiki-api.yandex-team.ru/_api/imain/content"
        %%

        GET-параметры:

        * widget_types: list – список строковых значений типов виджетов (опциональный)
        * ignore_cache: bool – флаг, определяющий использование кэша для выгрузки данных,
        значение по-умолчанию - false (опциональный)

        Ответ
        %%(js)
        "data": {
            "content": {
                "service_list": {
                    "data": {
                        "links": [
                            {
                                "id": "wiki",
                                "name": "Вики",
                                "url": "https://wiki.yandex-team.ru/~",
                                "icon": "https://wiki.yandex-team.ru/main/services/.files/wiki.svg"
                            },
                            {
                                "id": "tracker",
                                "name": "Трекер",
                                "url": "https://st.yandex-team.ru",
                                "icon": "https://wiki.yandex-team.ru/main/services/.files/tracker.svg"
                            },
                            {
                                "id": "staff",
                                "name": "Стафф",
                                "url": "https://staff.yandex-team.ru",
                                "icon": "https://wiki.yandex-team.ru/main/services/.files/staff.svg"
                            },
                            ...
                        ],
                    },
                    "settings": {}
                },
                "important_info": {},
                "about_company": {},
                "yandex_magazine": {},
                "offices": {},
                "contacts": {},
                "atushka_top": {}
            }
        }
        %%
        """
        serializer = self.get_serializer(data=request.GET)
        serializer.is_valid(raise_exception=True)
        request_widgets = serializer.data.get('widget_types')
        ignore_cache = serializer.data.get('ignore_cache')

        widget_types = sorted(CONTENT_WIDGET_ORDERING, key=CONTENT_WIDGET_ORDERING.get)
        if request_widgets:
            try:
                widget_types = sorted(
                    [WidgetTypes(q) for q in request_widgets], key=lambda x: CONTENT_WIDGET_ORDERING[x]
                )
            except ValueError as err:
                # нам что-то не то передали
                logger.warning(f'Unknown widget type in request data: {err}')

        data = {}
        for widget_type in widget_types:
            business_logic = WIDGET_REGISTRY.get(widget_type)
            if business_logic:
                data[widget_type.value] = business_logic.response_builder(
                    data=business_logic.data_class.from_dict(
                        bunker_datasource.get_widget_data(business_logic.bunker_xpath, ignore_cache)
                    ),
                    settings=business_logic.get_user_settings(user=request.user),
                    user=request.user,
                    lang=request.LANGUAGE_CODE[:2],
                )
            else:
                # этот тип виджета еще не реализован
                data[widget_type.value] = {}

        return Response(json.loads(json.dumps({'content': data}, default=lambda obj: obj.__dict__)))
