import json

from rest_framework.response import Response
from rest_framework.serializers import Serializer

from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.io import ok_response
from wiki.morda.consts import WidgetTypes
from wiki.morda.errors import JsonValidationError
from wiki.morda.datasources import bunker_datasource
from wiki.morda.widgets.registry import WIDGET_REGISTRY
from wiki.morda.widgets.service_list.widget import ServiceListWidget
from wiki.morda.serializers import ServicesOrderingSerializer
from wiki.morda.views.base import MainPageAPIView


class ServicesListView(MainPageAPIView):
    serializer_class = Serializer

    @raises(JsonValidationError)
    def get(self, request, *args, **kwargs):
        """
        Получить список сервисов для текущего пользователя.
        Если пользователь прежде настраивал индивидуальный порядок сортировки,
        то список будет отсортирован согласно настроек текущего пользователя.
        В противном случае будет применена сортировка по умолчанию (как на странице настроек для списка сервисов)

        %%
        curl -H "Authorization: OAuth <token>" "https://wiki-api.yandex-team.ru/_api/imain/services"
        %%

        Ответ
        %%(js)
        "data": {
            "data": {
                "links": [
                    {
                        "id": "wiki",
                        "name": "Вики",
                        "url": "https://wiki.yandex-team.ru/~",
                        "icon": "https://wiki.yandex-team.ru/main/services/.files/wiki.svg"
                    },
                    {
                        "id": "tracker",
                        "name": "Трекер",
                        "url": "https://st.yandex-team.ru",
                        "icon": "https://wiki.yandex-team.ru/main/services/.files/tracker.svg"
                    },
                    {
                        "id": "staff",
                        "name": "Стафф",
                        "url": "https://staff.yandex-team.ru",
                        "icon": "https://wiki.yandex-team.ru/main/services/.files/staff.svg"
                    },
                    {
                        "id": "mail",
                        "name": "Почта",
                        "url": "https://mail.yandex-team.ru",
                        "icon": "https://wiki.yandex-team.ru/main/services/.files/mail.svg"
                    },
                    ...
                ],
            settings: {...}
            }
        }
        %%
        """
        business_logic = WIDGET_REGISTRY[WidgetTypes.SERVICE_LIST]
        data = business_logic.response_builder(
            data=business_logic.data_class.from_dict(bunker_datasource.get_widget_data(business_logic.bunker_xpath)),
            settings=business_logic.get_user_settings(user=request.user),
            user=request.user,
            lang=request.LANGUAGE_CODE[:2],
        )

        return Response(json.loads(json.dumps(data, default=lambda obj: obj.__dict__)))


class ServicesOrderView(MainPageAPIView):
    serializer_class = ServicesOrderingSerializer

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить порядок сортировки списка сервисов для текущего пользователя.

        %%
        curl -H "Authorization: OAuth <token>" "https://wiki-api.yandex-team.ru/_api/imain/services/.order"
        %%

        Ответ
        %%(js)
        "data": {
            "ordering": [
                "wiki",
                "tracker",
                "staff"
            ]
        }
        %%
        """
        ordering = ServiceListWidget.get_user_settings(request.user).ordering
        return Response({'ordering': ordering})

    @raises()
    def post(self, request, *args, **kwargs):
        """
        Изменить порядок сортировки списка сервисов для текущего пользователя.

        %%
        curl -H "Authorization: OAuth <token>" -XPOST "https://wiki-api.yandex-team.ru/_api/imain/services/.order"
        %%
        Тело запроса:
        %%(js)
        {
            "ordering": [
                "wiki",
                "tracker",
                "staff"
            ]
        }
        %%

        Возвращает статус-код ответа.
        """
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        serializer.save()
        return ok_response()
