from abc import abstractmethod
from dataclasses import asdict
from typing import Dict, TypeVar, Generic

from wiki.morda.consts import WidgetTypes, ContainerTypes, CONTENT_WIDGET_ORDERING
from wiki.morda.models import PersonalizedWidget
from wiki.morda.widgets.base_entities import WidgetSettings, WidgetData
from wiki.users.models import User

DT = TypeVar('DT', bound=WidgetData)
ST = TypeVar('ST', bound=WidgetSettings)


class BaseWidget(Generic[DT, ST]):
    data_class: DT
    custom_settings_class: ST
    bunker_xpath: str

    widget_type: WidgetTypes
    container_type: ContainerTypes

    @abstractmethod
    def response_builder(self, data: DT, settings: ST, user: User, lang: str) -> Dict[str, any]:
        pass

    @classmethod
    def get_user_widget_settings(cls, user: User) -> dict:
        try:
            widget = PersonalizedWidget.objects.get(user=user, type=cls.widget_type, container_type=cls.container_type)
            return widget.custom_settings or {}
        except PersonalizedWidget.DoesNotExist:
            return {}

    @classmethod
    def save_user_widget_settings(
        cls,
        user: User,
        settings: ST,
    ):
        try:
            widget = PersonalizedWidget.objects.get(user=user, type=cls.widget_type, container_type=cls.container_type)
        except PersonalizedWidget.DoesNotExist:
            widget = PersonalizedWidget(
                user=user,
                type=cls.widget_type,
                container_type=cls.container_type,
                is_visible=True,
                order=CONTENT_WIDGET_ORDERING[cls.widget_type],
                custom_settings={},
            )

        widget.custom_settings = asdict(settings)
        widget.save()
