
import logging
import traceback

from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext
from wiki.legacy.env import environment_type

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, supply_events
from wiki.org import org_group, org_user
from wiki.pages.models import AccessRequest
from wiki.users.models import Group

log = logging.getLogger(__name__)


class AccessResolutionGen(BaseGen):
    @supply_events(EventTypes.resolve_access)
    def generate(self, events, settings):
        if events:
            processed_receivers = []
            page = events[0].page
            for event in events:
                try:
                    ar = AccessRequest.objects.get(pk=event.meta['access_request_id'])
                    if 'notify_all' in event.meta:
                        notify_applicant = not event.meta['notify_all']
                    else:
                        notify_applicant = False
                    if notify_applicant:
                        receivers_list = [ar.applicant]
                    else:
                        receivers_list = self.users_from_meta(event)
                    reason = ar.verdict_reason
                except Exception:
                    log.exception('Access resolution event handling error')
                    reason = None
                    ar = None
                    receivers_list = self.users_from_meta(event)

                event_type = event.meta['resolution_type']
                for receiver in receivers_list:
                    if event_type != 'notchanged' and not page.has_access(receiver):
                        continue
                    if receiver.pk not in processed_receivers and receiver != event.author:
                        processed_receivers.append(receiver.pk)
                        _ = self.email_language(receiver, strict_mode=True)
                        if _ is None:
                            continue
                        email, name, lang = _
                        translation.activate(lang)
                        if event_type == 'notchanged':
                            title = ugettext('Access forbidden: to %(wiki_entity)s %(cluster)s titled %(title)s')
                        else:
                            title = ugettext('Access granted: to %(wiki_entity)s %(cluster)s titled %(title)s')
                        title = title % {
                            'cluster': self.page_name_for_print(page),
                            'title': self.page_title_for_print(page),
                            'wiki_entity': page.dative,
                        }

                        params = self.default_params.copy()
                        params.update(
                            {
                                'page_url': page.absolute_url,
                                'page_title': self.page_title_for_print(page),
                                'event': event,
                                'access_request': ar,
                                'receiver': receiver,
                            }
                        )
                        if 'group_id' in event.meta:
                            try:
                                params['group'] = org_group().get(id=event.meta['group_id'])
                            except Group.DoesNotExist:
                                pass
                        if reason:
                            params['reason'] = reason
                        self.add_chunk(
                            EmailDetails._replace(
                                receiver_email=email,
                                receiver_name=name,
                                receiver_lang=lang,
                                subject=title,
                                author_name=event.author.staff.inflections.subjective,
                                reply_to=event.author.staff.get_email(),
                            ),
                            render_to_string('access_resolution_%s.html' % event_type, params),
                        )
        return self.reply

    def users_from_meta(self, event):
        """Return list of django Users according to meta

        TODO: fix
        Does not process situation with access object having is_common==True
        (this is === notify every user)
        """
        meta = event.meta
        if 'receiver_id' in meta:
            # receiver_id присутствует в мете только в случае отказа в доступе
            # к странице. В этом поле id из таблицы auth_user, так что искать
            # нужно в ней.
            return org_user().filter(id=meta['receiver_id'][0]).select_related('staff')
        if 'is_owner' in meta and meta['is_owner']:
            return (
                org_user()
                .filter(username__in=[author.username for author in event.page.get_authors()])
                .select_related('staff')
            )
        if 'staff_id' in meta and meta['staff_id']:
            return org_user().filter(staff__id=meta['staff_id']).select_related('staff')
        if 'user_id' in meta and meta['user_id']:
            return org_user().filter(pk=meta['user_id']).select_related('staff')
        if 'group_id' in meta and meta['group_id']:
            group_id = meta['group_id']
            try:
                group = org_group().get(pk=group_id)

            except Group.DoesNotExist:
                print('Somehow I could not get Group with id', group_id, 'meta was', meta, 'id of event was', event.pk)
                print(traceback.format_exc())
                if environment_type != 'production':
                    raise
            else:
                return group.get_all_members()
