
import logging

from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, remove_watch, supply_events
from wiki.org import org_user
from wiki.pages.logic import subscription
from wiki.pages.models import Comment

logger = logging.getLogger(__name__)


class AddCommentGen(BaseGen):
    @supply_events(EventTypes.add_comment)
    def generate(self, events, settings):
        User = self._get_user_model()
        if events:
            notify_watchers = set(subscription.get_united_subscribed_users(events[0].page))
            for event in events:
                try:
                    comment = Comment.objects.get(pk=event.meta['comment_id'])
                except Comment.DoesNotExist:
                    continue
                is_answer = False
                notify_users = notify_watchers.copy()

                # будет ли среди писем ответ автору родительского комментария
                if comment.parent_id and comment.parent.user_id != comment.user_id:
                    try:
                        user = org_user().get(username=comment.parent.user.username)
                    except User.DoesNotExist:
                        logger.warning(
                            'No user with name "%s", notification will not be sent', comment.parent.user.username
                        )
                    else:
                        notify_users.add(user)
                        is_answer = True

                for user in notify_users:
                    if not event.page.has_access(user):
                        remove_watch(user, event.page)
                        continue
                    if event.author != user:

                        tmp = self.email_language(user, strict_mode=True)
                        if tmp is None:
                            continue

                        email, rname, language = tmp
                        translation.activate(language)
                        params = self.default_params.copy()
                        params.update(
                            {
                                'page_title': self.page_title_for_print(event.page),
                                'page_url': event.page.absolute_url + '?show_comments=1',
                                'comment': comment,
                                'comment_body': comment.body.replace('&nbsp;', ' '),
                                'event': event,
                                'as_answer': False,
                            }
                        )
                        if is_answer and comment.parent.user == user.username:
                            params.update({'as_answer': True})
                        self.add_chunk(
                            EmailDetails._replace(
                                receiver_email=email,
                                receiver_name=rname,
                                receiver_lang=language,
                                subject=ugettext('Commented: page %(name)s titled %(title)s')
                                % {
                                    'name': self.page_name_for_print(event.page),
                                    'title': self.page_title_for_print(event.page),
                                },
                                author_name=event.author.staff.inflections.subjective,
                                reply_to=event.author.staff.get_email(),
                            ),
                            render_to_string('addcomment.html', params),
                        )
        return self.reply
