
from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ungettext
from wiki.intranet.models import Staff

from wiki.files.models import File
from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, remove_watch, supply_events
from wiki.pages.logic import subscription


class AddFileGen(BaseGen):
    @supply_events(EventTypes.add_file)
    def generate(self, events, settings):
        if events:
            self._cached_files = self.files(events)
            e = events[0]
            self.one_of_events = e
            self.page = e.page
            watchers = subscription.get_united_subscribed_users(e.page)
            self.page_tag = self.page_name_for_print(e.page)
            for watcher in watchers:
                self.notify(watcher)
        return self.reply

    def files(self, events):
        events_ids = set([e.meta['id'] for e in events])
        files_ids = File.active.filter(id__in=events_ids).values_list('id', flat=True)

        result = []
        for event in events:
            deleted = True
            if event.meta['id'] in files_ids:
                deleted = False
            result.append(
                {
                    'event': event,
                    'url': event.meta['url'],
                    'size': event.meta['size'],
                    'name': event.meta['filename'],
                    'deleted': deleted,
                }
            )
        return result

    def notify(self, user):
        if getattr(user.staff, 'is_dismissed', False):
            return
        if not (self.page.has_access(user)):
            remove_watch(user, self.page)
            return
        tmp = self.email_language(user, strict_mode=True)
        if tmp is None:
            return
        email, rname, language = tmp
        translation.activate(language)
        attached_files = []
        authors = set()
        for file in self._cached_files:
            if file['event'].author != user:
                attached_files.append(file)
                authors.add(file['event'].author)
        if not attached_files:
            return
        params = self.default_params.copy()
        params.update(
            {
                'files': attached_files,
                'page_tag': file['event'].page.tag,
                'page_title': self.page_title_for_print(self.page),
                'page_url': self.page.absolute_url,
            }
        )
        subject = ungettext(
            'Attached file: to page %(name)s titled %(title)s',
            'Attached files: to page %(name)s titled %(title)s',
            len(attached_files),
        )
        subject = subject % {
            'name': self.page_tag,
            'title': self.page_title_for_print(self.one_of_events.page),
        }

        if len(authors) == 1:
            author = authors.pop()

            username, reply_to_email = get_email_name(author)

            receiver_info = EmailDetails._replace(
                receiver_email=email,
                receiver_name=rname,
                receiver_lang=language,
                subject=subject,
                author_name=username,
                reply_to=reply_to_email,
            )
        else:
            receiver_info = EmailDetails._replace(
                receiver_email=email, receiver_name=rname, receiver_lang=language, subject=subject
            )
        self.add_chunk(receiver_info, render_to_string('addfile.html', params))


def get_email_name(django_user):
    """
    Вернуть email и имя пользователя, независимо от того, робот он или живой.

    User -> (string, string)

    @type django_user: django.contrib.auth.models.User
    @rtype: tuple
    """

    try:
        staff = django_user.staff
    except Staff.DoesNotExist:
        return (
            django_user.username,
            django_user.email,
        )

    return (
        staff.inflections.subjective,
        staff.get_email(),
    )
