import logging
from collections import defaultdict, namedtuple

import pytz
from django.conf import settings
from django.contrib.auth import get_user_model
from django.utils import timezone, translation

from wiki.intranet.models import Staff
from wiki.legacy.quotation import html_quotation
from wiki.legacy.translit import name_translit
from wiki.pages.models import PageWatch
from wiki.users.core import get_support_contact
from wiki.utils.timezone import select_timezone

logger = logging.getLogger('generators')

event_types = '''create delete edit access request_access keywords rename
    add_file delete_file add_comment delete_comment watch unwatch
    resolve_access closed_page_access change_owner move_cluster
    remove_comment request_ownership change_to_redirect revert_from_redirect
    mark_obsolete mark_actual subscribe_other_user change_authors request_page_author
    add_author change_access enable_page_readonly disable_page_readonly change_position
    '''.split()
EventTypes = namedtuple('EventTypes', ' '.join(event_types))._make(list(range(1, len(event_types) + 1)))
IS_INTRANET = getattr(settings, 'IS_INTRANET', False)
SETTINGS = {}
for etype in EventTypes:
    SETTINGS[etype] = {}

# здесь можно собрать дефолтные параметры для ивентов, чтобы не писать иx
# в миллионе мест
DEFAULT_EVENTS_PARAMS = {
    ('page', EventTypes.create): {
        'timeout': 5,
        'notify': True,
    },
    ('page', EventTypes.edit): {
        'timeout': 20,
        'notify': True,
    },
    ('page', EventTypes.keywords): {
        'timeout': 20,
        'notify': True,
    },
    ('page', EventTypes.change_to_redirect): {
        'notify': False,
    },
    ('page', EventTypes.revert_from_redirect): {
        'notify': False,
    },
    ('page', EventTypes.add_comment): {
        'timeout': 0,
        'notify': True,
    },
}

SETTINGS[EventTypes.edit] = {'generate': True}
SETTINGS[EventTypes.request_access] = {'generate': True, 'suggest': True}
SETTINGS[EventTypes.resolve_access] = {'generate': True}

NO_GENERATE = 'no_generate'

translit_offset = ord('а')
translit_transtab = [
    'a',
    'b',
    'v',
    'g',
    'd',
    'e',
    'zh',
    'z',
    'i',
    'y',
    'k',
    'l',
    'm',
    'n',
    'o',
    'p',
    'r',
    's',
    't',
    'u',
    'f',
    'kh',
    'ts',
    'ch',
    'sh',
    'shch',
    '',
    'i',
    '',
    'e',
    'ju',
    'ya',
]

email_details = 'receiver_email receiver_name receiver_lang subject author_name reply_to cc bcc'
EmailDetails = namedtuple('EmailDetails', email_details)(*([None] * len(email_details.split())))


def translit(st):
    res = []
    if not isinstance(st, str):
        st = str(st, 'utf-8')
    for c in st.lower():
        tabpos = ord(c) - translit_offset
        if 0 <= tabpos < 32:
            res.append(translit_transtab[tabpos])
        elif c != 'ё':
            res.append(c)
        else:
            res.append('yo')
    return ''.join(res)


def supply_events(marks):
    """
    Filter out events whose type not in {marks}
    """
    if isinstance(marks, str) or not hasattr(marks, '__iter__'):
        marks = [marks]

    def _supply(fn):
        def new(self, events, settings=None):
            if settings is None:
                settings = {}
            self.reply = defaultdict(list)
            events = [e for e in events if e.event_type in marks]
            if events:
                return fn(self, events, settings)
            return self.reply

        return new

    return _supply


lazy_page_title = {}


class BaseGen(object):
    default_params = {'is_intranet': IS_INTRANET}

    def __init__(self):
        self.reply = defaultdict(list)

    def email_language(self, user, strict_mode=False):
        """
        Return user's email, language and name or send default data.
        If strict_mode=True method will return None if user is dismissed, is robot or does not exist

        @type user: wiki.users.models.User
        @param strict_mode: return None if user is dismissed, is robot or does not exist

        @rtype: tuple
        @return: base64-encoded email, username and language
        """
        user_data = {}

        if not isinstance(user, get_user_model()):
            # not django-user for some reason
            logger.warning('expected django.contrib.auth.model.User, got %s %s instead', str(type(user)), str(user))
            return self._generate_email_w_lang(**user_data)

        if user.is_robot():
            # robot
            if strict_mode:
                return None

        elif getattr(user.staff, 'is_dismissed', False):
            # dismissed employee
            if strict_mode:
                return None

            user_data.update(
                {'name': 'dismissed ' + self._get_username(user.staff), 'language': user.staff.lang_ui or 'en'}
            )

        else:
            # active user
            try:
                user_data['email'] = self._get_user_email(user.staff)
            except AttributeError as exc:
                logger.warning(exc, exc_info=True)
                if strict_mode:
                    return None

            user_data.update({'name': self._get_username(user.staff), 'language': user.staff.lang_ui or 'en'})

        if not user.is_robot():
            timezone.activate(pytz.timezone(select_timezone(user)))

        return self._generate_email_w_lang(**user_data)

    def _get_user_model(self):
        return get_user_model()

    def _get_user_email(self, staff):
        """
        Return wiki-user email from staff

        @raise: AttributeError if email on staff doesn't exist
        """
        assert isinstance(staff, Staff), 'staff parameter must be of Staff type'

        real_email = staff.get_email()
        if real_email:
            return real_email

        raise AttributeError('user staff_id=%s has no email', str(staff.id))

    def _get_username(self, staff):
        """
        Return wiki-user language-aware username representation from staff
        """
        assert isinstance(staff, Staff), 'staff parameter must be of Staff type'

        language = staff.lang_ui

        if language == 'en':
            first_name = staff.first_name_en or name_translit(staff.first_name)
            last_name = staff.last_name_en or name_translit(staff.last_name)
        else:
            first_name = staff.first_name
            last_name = staff.last_name

        """
          [EBI-1004] В федеративных пользователях если имя и-или фамилия не заполнены,
          происходит фоллбек до логина, который почта на домене

          письмо с адресатом neofelis@somedomain.ru neofelis@somedomain.ru <neofelis@somedomain.ru>
          не парсится и отправка падает;

          пусть в таких случаях будет neofelis <neofelis@somedomain.ru>
        """
        parts = [section for section in [first_name, last_name] if '@' not in section]

        if len(parts) == 0:
            return staff.login.split('@')[0]

        return ' '.join(parts)

    def _generate_email_w_lang(self, name=None, email=settings.SUPPORT_EMAIL, language='ru'):
        """
        Return base64-encoded email, username and language
        """
        if not name:
            name = get_support_contact()
        email = f'{name} <{email}>'
        return email, name, language

    def add_chunk(self, email_username, chunk):
        self.reply[email_username].append(chunk)

    def page_name_for_print(self, page=None, page_tag=None):
        if page is not None:
            page_tag = page.tag
            if page.status == 0:  # удалена
                page_tag = page.supertag_before_deletion

        if page_tag.count('/') > 2:
            spl = page_tag.split('/')
            return '/' + '/'.join(spl[:1] + ['...'] + spl[-2:])
        return '/' + page_tag

    def page_title_for_print(self, page):
        if page.title:
            result = html_quotation(page.title)
        result = html_quotation(page.tag.split('/')[-1])
        return result

    def user_name_for_print(self, user):
        try:
            if user.staff:
                name = ' '.join((user.staff.first_name, user.staff.last_name))
            else:
                name = user.username
            language = translation.get_language()
            if language == 'en':
                name = translit(name)
        except Staff.DoesNotExist:
            return user.username

    def get_page_authors_emails(self, page):
        """
        Возвращает уникальный список таплов с email адресами авторов страницы - (email, receiver_name, language).
        Если все авторы страницы - уволенные сотрудники, то возвращает SUPPORT_EMAIL адрес
        или адрес администратора организации (только для бизнес вики)
        """
        result_list = list()
        if page.is_all_authors_dismissed():
            if settings.IS_BUSINESS:
                from wiki.users_biz.dao import get_admin_group_members

                receivers = get_admin_group_members()
            else:
                return [(settings.SUPPORT_EMAIL, get_support_contact(), 'ru')]
        else:
            receivers = page.get_authors()

        for receiver in receivers:
            email_tuple = self.email_language(receiver, strict_mode=True)
            if email_tuple:
                result_list.append(email_tuple)

        return result_list


def remove_watch(user, page):
    """Удалить подписку.

    DjangoUser, Page -> None
    """
    PageWatch.objects.filter(page=page, user=user.username).delete()
