import logging

from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, supply_events
from wiki.pages.logic import url_to_wiki_page_in_frontend
from wiki.pages.logic.subscription import is_subscribed_to_page

logger = logging.getLogger(__name__)


class ChangeAccessGen(BaseGen):
    @supply_events(EventTypes.change_access)
    def generate(self, events, settings):
        for event in events:
            old_access_type = event.meta['old_access_type']
            new_access_type = event.meta['new_access_type']
            added_users = event.meta.get('added_users', [])
            removed_users = event.meta.get('removed_users', [])
            added_groups = event.meta.get('added_groups', [])
            removed_groups = event.meta.get('removed_groups', [])

            # Translators:
            #   ru: Изменен доступ: %(entity)s %(address)s с именем %(title)s
            #   en: Changed access: %(entity)s %(address)s titled %(title)s
            subject = 'Changed access: %(entity)s %(address)s titled %(title)s'

            for author in event.page.get_authors():
                if author == event.author or not is_subscribed_to_page(author, event.page):
                    continue

                email_tuple = self.email_language(author, strict_mode=True)
                if email_tuple is None:
                    continue

                receiver_email, receiver_name, receiver_lang = email_tuple
                translation.activate(receiver_lang)

                email_subject = ugettext(subject) % {
                    'entity': event.page.subjective,
                    'address': self.page_name_for_print(event.page),
                    'title': self.page_title_for_print(event.page),
                }

                email_details = EmailDetails._replace(
                    receiver_email=receiver_email,
                    receiver_name=receiver_name,
                    receiver_lang=receiver_lang,
                    subject=email_subject,
                    author_name=event.author.staff.inflections.subjective,
                    reply_to=event.author.staff.get_email(),
                )

                message = render_to_string(
                    'change_access.html',
                    {
                        'event': event,
                        'page_title': self.page_title_for_print(event.page),
                        'old_access_type': old_access_type,
                        'new_access_type': new_access_type,
                        'added_users': added_users,
                        'removed_users': removed_users,
                        'added_groups': added_groups,
                        'removed_groups': removed_groups,
                        'unsubscribe_url': url_to_wiki_page_in_frontend(event.page, action='unwatch'),
                    },
                )
                self.add_chunk(email_details, message)

        return self.reply
