import logging

from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, supply_events
from wiki.org import org_user
from wiki.pages.logic import url_to_wiki_page_in_frontend
from wiki.pages.logic.subscription import is_subscribed_to_page

logger = logging.getLogger(__name__)


class ChangeAuthorGen(BaseGen):
    @supply_events([EventTypes.add_author, EventTypes.change_authors])
    def generate(self, events, settings):
        for event in events:
            if event.event_type == EventTypes.add_author:
                new_author_name = event.meta['new_author']

                # Translators:
                #   ru: Добавлен автор: %(entity)s %(address)s с именем %(title)s
                #   en: Added author: %(entity)s %(address)s titled %(title)s
                subject = 'Added author: %(entity)s %(address)s titled %(title)s'

                self.process_event(
                    event=event,
                    author_name=new_author_name,
                    subject=subject,
                    template_name='added_author.html',
                    by_request=True,
                )
            else:
                with_subpages = event.meta['with_subpages']
                added_authors = event.meta.get('added_authors', [])
                removed_authors = event.meta.get('removed_authors', [])
                pages_count = event.meta.get('pages_count', 0)

                for added_author_name in added_authors:
                    # Translators:
                    #   ru: Добавлен автор: %(entity)s %(address)s с именем %(title)s
                    #   en: Added author: %(entity)s %(address)s titled %(title)s
                    subject = 'Added author: %(entity)s %(address)s titled %(title)s'

                    self.process_event(
                        event=event,
                        author_name=added_author_name,
                        subject=subject,
                        template_name='added_author.html',
                        with_subpages=with_subpages,
                        pages_count=pages_count,
                    )

                for removed_author_name in removed_authors:
                    # Translators:
                    #   ru: Удален автор: %(entity)s %(address)s с именем %(title)s
                    #   en: Removed author: %(entity)s %(address)s titled %(title)s
                    subject = 'Removed author: %(entity)s %(address)s titled %(title)s'

                    self.process_event(
                        event=event,
                        author_name=removed_author_name,
                        subject=subject,
                        template_name='removed_author.html',
                        with_subpages=with_subpages,
                        pages_count=pages_count,
                    )

        return self.reply

    def process_event(
        self, event, author_name, subject, template_name, with_subpages=False, pages_count=0, by_request=False
    ):
        try:
            user = org_user().get(username=author_name)
        except self._get_user_model().DoesNotExist:
            logger.warning('No user with name "%s", notification will not be sent', author_name)
            return

        if user == event.author or not is_subscribed_to_page(user, event.page):
            return

        email_tuple = self.email_language(user, strict_mode=True)
        if email_tuple is None:
            return

        receiver_email, receiver_name, receiver_lang = email_tuple
        translation.activate(receiver_lang)

        email_subject = ugettext(subject) % {
            'entity': ugettext('cluster') if with_subpages else event.page.subjective,
            'address': self.page_name_for_print(event.page),
            'title': self.page_title_for_print(event.page),
        }

        email_details = EmailDetails._replace(
            receiver_email=receiver_email,
            receiver_name=receiver_name,
            receiver_lang=receiver_lang,
            subject=email_subject,
            author_name=event.author.staff.inflections.subjective,
            reply_to=event.author.staff.get_email(),
        )

        message = render_to_string(
            template_name,
            {
                'event': event,
                'by_request': by_request,
                'with_subpages': with_subpages,
                'pages_count': pages_count,
                'page_title': self.page_title_for_print(event.page),
                'unsubscribe_url': url_to_wiki_page_in_frontend(event.page, action='unwatch'),
            },
        )
        self.add_chunk(email_details, message)
