import datetime
from json import loads

from django.conf import settings
from django.template.loader import render_to_string
from django.utils import translation
from django.utils.html import escape
from django.utils.translation import ugettext as _

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, remove_watch, supply_events
from wiki.notifications.generators.utils import format_for_nice_datetime
from wiki.pages.logic import subscription
from wiki.pages.logic.urls import url_to_wiki_page_in_frontend
from wiki.pages.templatetags.evil_tags import declination_verb
from wiki.utils.diff import DifflibDiff

_diff = DifflibDiff()

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)
DEFAULT_TIMEOUT = datetime.timedelta(minutes=settings.GRID_PAGEEVENT_TIMEOUT)


def _handle_raw_row(titles, raw_row):
    """
    Приводит объект raw_row, полученный из PageEvent, к списку из значений в ячейках в правильном порядке.
    raw_row - либо list (старый формат, отсортированы по id ячеек), либо dict (новый формат, в качестве ключей -
    id ячеек).
    """
    if isinstance(raw_row, dict):
        result = []
        for id, title in titles:
            if id in raw_row:
                result.append(raw_row[id])
            else:
                result.append('')
        return result
    else:
        return raw_row
        # Вообще говоря, это неверно, потому что структура таблицы могла полностью поменяться, столбцы могут идти
        # не в том порядке, половина из них может быть замещена совершенно другими столбцами, и полученный дифф будет
        # совершенно некорректным. Но такая проблема была и раньше, и просуществует не более, чем 20 минут с момента
        # релиза этого кода (settings.GRID_PAGEEVENT_TIMEOUT), так что забиваем на это.


class EditionGen(BaseGen):
    @supply_events(EventTypes.edit)
    def generate(self, events, generator_settings):
        if not events:
            return self.reply
        event = events[0]
        page = event.page
        watchers = subscription.get_united_subscribed_users(page)

        titles = self.get_titles(grid=page)

        for user in watchers:
            authors = self.get_authors(events, [user.username])
            if not authors:
                continue

            if getattr(user.staff, 'is_dismissed', False):
                continue

            if settings.IS_BUSINESS and user.is_dir_robot:
                remove_watch(user, page)
                continue

            if not page.has_access(user):
                remove_watch(user, page)
                continue

            diff = self.get_diff(events=events, titles=titles, excluded_users=[user])
            additions, deletions, editions = diff

            params = self.default_params.copy()
            email_data = self.email_language(user, strict_mode=True)
            if email_data is None:
                continue
            email, receiver_name, language = email_data
            translation.activate(language)
            subject = _('Page edited: %(address)s titled %(title)s') % {
                'title': self.page_title_for_print(page),
                'address': self.page_name_for_print(page),
            }

            receiver_info = EmailDetails._replace(
                receiver_email=email, receiver_name=receiver_name, receiver_lang=language, subject=subject
            )

            if len(authors) == 1:
                for author in authors:
                    receiver_info = receiver_info._replace(
                        author_name=author.staff.inflections.subjective, reply_to=author.staff.get_email()
                    )
                    params['author'] = author
            suffix_for_editing = ''
            if language == 'ru':
                if len(authors) == 1:
                    for author in authors:
                        suffix_for_editing = declination_verb(author.staff)
                else:
                    suffix_for_editing = 'и'

            # Момент времени перед первым изменением, попадающим в дифф
            start_revision_time = events[-1].created_at - datetime.timedelta(minutes=1)

            # Хак, но иначе время последней правки, попавшей в дифф, не получить. Объекты Revision и PageEvent после
            # создания могут дозаписываться.
            end_revision_time = events[0].timeout - DEFAULT_TIMEOUT

            show_seconds = False
            if end_revision_time.minute != start_revision_time.minute:
                if (end_revision_time - start_revision_time).seconds < 60:
                    show_seconds = True
            params.update(
                {
                    'start_revision_time': start_revision_time,
                    'end_revision_time': end_revision_time,
                    'format_for_nice_datetime': format_for_nice_datetime(show_seconds),
                    'authors': authors,
                    'events': events,
                    'titles': titles,
                    'additions': additions,
                    'deletions': deletions,
                    'editions': editions,
                    'page_title': self.page_title_for_print(page),
                    'page': page,
                    'unsubscribe_url': url_to_wiki_page_in_frontend(page, action='unwatch'),
                    'suffix': suffix_for_editing,
                    'page_is_grid': True,
                }
            )
            self.add_chunk(receiver_info, render_to_string('edition.html', params))

        return self.reply

    def get_authors(self, events, exclude_logins):
        authors = set()
        for event in events:
            if event.author.username in exclude_logins:
                continue
            authors.add(event.author)
        return authors

    def get_titles(self, grid):
        return [(field['name'], field['title']) for field in loads(grid.body)['structure']['fields']]

    def get_diff(self, events, titles, excluded_users=None):
        excluded_users = excluded_users or []

        deletions, additions = {}, {}
        for event in events:
            if event.author in excluded_users:
                continue

            # События в обратном хронологическом порядке.
            # Для каждой строки нужно взять самое раннее удаление и
            # самое позднее добавление, чтобы дифф был корректным.
            if 'diff' in event.meta:
                for key, row in event.meta['diff']['deleted'].items():
                    deletions[key] = _handle_raw_row(titles, row)
                for key, row in event.meta['diff']['added'].items():
                    if key not in additions:
                        additions[key] = _handle_raw_row(titles, row)

        # В зависимости от формата столбца в ячейке могут оказаться
        # разные типы данных.
        # По идее нужно обрабатывать каждый отдельно (например,
        # вместо логинов показывать ссылки на стафф),
        # но пока реализовано просто.
        for dictionary in (deletions, additions):
            for key in dictionary:
                line = []
                for item in dictionary[key]:
                    if isinstance(item, list):
                        line.append(', '.join(str(i) for i in item))
                    else:
                        line.append(str(item))
                dictionary[key] = line

        editions = {}
        for key in list(additions.keys()):  # Не iterkeys, потому что будем находу удалять ключи
            if key in deletions:
                editions[key] = []
                for old_item, new_item in zip(deletions[key], additions[key]):
                    diff = _diff.diff_texts(escape(old_item), escape(new_item))
                    diff = (
                        diff.replace('\n', '<br/>')
                        .replace('<del>', '<span style="color: #de1822; text-decoration: line-through;">')
                        .replace('<ins>', '<span style="color: #45b63f;">')
                        .replace('</del>', '</span>')
                        .replace('</ins>', '</span>')
                    )
                    editions[key].append(diff)
                del additions[key]
                del deletions[key]

        return additions, deletions, editions
