
from django.conf import settings as global_settings
from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext
from wiki.legacy.env import environment_type

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, supply_events
from wiki.pages.models import Comment

IS_INTRANET = getattr(global_settings, 'IS_INTRANET', False)
if IS_INTRANET:
    from wiki.intranet.models import GroupMembership
    from wiki.pages.dao.officiality import get_officiality


class MarkActualityGen(BaseGen):
    @supply_events([EventTypes.mark_actual, EventTypes.mark_obsolete])
    def generate(self, events, settings):
        if events:
            for event in events:
                page = event.page

                params = self.default_params.copy()
                params.update(
                    {'event': event, 'page_tag': page.tag, 'is_actual': event.event_type == EventTypes.mark_actual}
                )

                comment_id = event.meta.get('comment_id')
                if comment_id:
                    try:
                        comment = Comment.objects.get(pk=comment_id)
                        params['comment'] = comment.body
                    except Comment.DoesNotExist:
                        if environment_type != 'production':
                            raise

                processed_receivers = [event.author.staff if IS_INTRANET else event.author]
                for author in page.get_authors():
                    self._generate_for_receiver(event, author, params, processed_receivers)

                if not IS_INTRANET or not page.is_official:
                    continue

                officiality = get_officiality(page.pk)
                if officiality is None:
                    if environment_type != 'production':
                        raise Exception('Officiality not found for official page (pk=%s)' % page.pk)
                    continue

                for person in officiality.responsible_persons.all():
                    self._generate_for_receiver(event, person, params, processed_receivers)

                groups_pks = [g.pk for g in officiality.responsible_groups.all()]
                members = self.get_groups_all_members(groups_pks)
                for member in members:
                    self._generate_for_receiver(event, member.staff.user, params, processed_receivers)

        return self.reply

    def _generate_for_receiver(self, event, receiver, params, processed_receivers):
        if IS_INTRANET:
            if receiver.staff.is_dismissed or receiver.staff in processed_receivers:
                return
            processed_receivers.append(receiver.staff)
        else:
            if receiver in processed_receivers:
                return
            # в случае не-интранета будет только один вызов для владельца, можно не добавлять в processed_receivers

        email_tuple = self.email_language(receiver, strict_mode=True)
        if email_tuple is None:
            return

        email, receiver_name, language = email_tuple
        translation.activate(language)

        subject = ugettext('Marked %(mark)s: page %(name)s titled %(title)s') % {
            'mark': ugettext('actual' if event.event_type == EventTypes.mark_actual else 'obsolete'),
            'title': self.page_title_for_print(event.page),
            'name': self.page_name_for_print(event.page),
        }

        self.add_chunk(
            EmailDetails._replace(
                receiver_email=email,
                receiver_name=receiver_name,
                receiver_lang=language,
                subject=subject,
                author_name=event.author.staff.inflections.subjective,
                reply_to=event.author.staff.get_email(),
            ),
            render_to_string('mark_actuality.html', params),
        )

    @staticmethod
    def get_groups_all_members(groups_pks):
        return GroupMembership.objects.filter(group__pk__in=groups_pks).select_related('staff')
