from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, remove_watch, supply_events
from wiki.pages.logic import subscription
from wiki.pages.models import Page


class MoveGen(BaseGen):
    @supply_events([EventTypes.move_cluster, EventTypes.create])
    def generate(self, events, settings):
        """Notify only about the last one"""
        if any(_.event_type == EventTypes.create for _ in events):
            return self.reply
        events = [e for e in events if e.event_type == EventTypes.move_cluster]
        if events:
            e = events[0]
            if not e.page:
                return self.reply

            ancestors = list(e.page.ancestors)
            if ancestors:
                subscribers = subscription.get_united_subscribed_users(page=ancestors[0])
                users_to_notify = [user for user in subscribers if user != e.author]
            else:
                return self.reply

            params = self.default_params.copy()
            params.update(
                {
                    'event': e,
                    'with_children': e.meta.get('with_children', False),
                    'page_title': self.page_title_for_print(e.page),
                }
            )

            for user in users_to_notify:
                if not e.page.has_access(user):
                    remove_watch(user, e.page)
                    continue

                lang_tuple = self.email_language(user, strict_mode=True)
                if lang_tuple is None:
                    continue

                email, receiver_name, language = lang_tuple

                translation.activate(language)
                subject = translation.ugettext('Cluster moved: %(cluster)s titled %(title)s') % {
                    'cluster': self.page_name_for_print(e.page),
                    'title': self.page_title_for_print(e.page),
                }

                params['is_owner'] = True if user.username in e.page.get_authors() else False
                if e.page.page_type == Page.TYPES.GRID:
                    moved_your_entities = ugettext('notify.move:Your grid word')
                else:
                    moved_your_entities = ugettext('notify.move:Your page word')
                params['moved_your_entities'] = moved_your_entities
                if params['with_children']:
                    if e.page.page_type == Page.TYPES.GRID:
                        params['your_page_pronoun'] = ugettext('notify.generators.place of Your grid')
                    else:
                        params['your_page_pronoun'] = ugettext('notify.generators.place of Your page')
                else:
                    params['your_page_pronoun'] = ''
                self.add_chunk(
                    EmailDetails._replace(
                        receiver_email=email,
                        receiver_name=receiver_name,
                        receiver_lang=language,
                        subject=subject,
                        author_name=e.author.staff.inflections.subjective,
                        reply_to=e.author.staff.get_email(),
                    ),
                    render_to_string('move_cluster.html', params),
                )
        return self.reply
