import logging

from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext
from wiki.legacy.env import environment_type

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, remove_watch, supply_events
from wiki.org import org_user

logger = logging.getLogger(__name__)


class ChangeOwnerGen(BaseGen):
    @supply_events(EventTypes.change_owner)
    def generate(self, events, settings):
        """
        Notify only about the last one
        """
        User = self._get_user_model()
        if events:
            e = events[0]
            meta = e.meta
            user = None
            try:
                user = org_user().get(username=meta.get('owner_name', ''))
            except User.DoesNotExist:
                if 'owner_name' in meta:
                    logger.exception(
                        'Can\'t find page owner with name "%s", PageEvent (%d)', meta.get('owner_name', ''), e.pk
                    )
                else:
                    logger.exception('PageEvent %s', e.pk)
                if environment_type != 'production':
                    raise
            previous_owner = None
            if e.author.username != meta.get('previous_owner'):
                try:
                    previous_owner = org_user().select_related('staff').get(username=meta.get('previous_owner'))
                except User.DoesNotExist:
                    pass
            params = self.default_params.copy()
            params.update(
                {
                    'new_owner': user,
                    'previous_owner': previous_owner or e.author,
                    'with_children': meta.get('with_children', False),
                    'page_title': self.page_title_for_print(e.page),
                    'event': e,
                }
            )
            if user != e.author:
                if not e.page.has_access(user):
                    remove_watch(user, e.page)
                else:
                    # notify new owner
                    tmp = self.email_language(user, strict_mode=True)
                    if tmp is not None:
                        email, receiver_name, language = tmp
                        translation.activate(language)
                        subject = ugettext("Owner of %(handler)s changed: %(cluster)s titled \"%(title)s\"") % {
                            'cluster': self.page_name_for_print(e.page),
                            'title': self.page_title_for_print(e.page),
                            'handler': e.page.genitive,
                        }
                        self.add_chunk(
                            EmailDetails._replace(
                                receiver_email=email,
                                receiver_name=receiver_name,
                                receiver_lang=language,
                                subject=subject,
                                author_name=e.author.staff.inflections.subjective,
                                reply_to=e.author.staff.get_email(),
                            ),
                            render_to_string('became_owner.html', params),
                        )
            if e.author != previous_owner and previous_owner:
                if getattr(previous_owner.staff, 'is_dismissed', False):
                    return self.reply
                if not e.page.has_access(previous_owner):
                    remove_watch(previous_owner, e.page)
                    return self.reply

                tmp = self.email_language(previous_owner, strict_mode=True)
                if tmp is None:
                    return self.reply

                email, receiver_name, language = tmp
                translation.activate(language)
                subject = ugettext('Page owner changed: %(cluster)s titled %(title)s') % {
                    'cluster': self.page_name_for_print(e.page),
                    'title': self.page_title_for_print(e.page),
                }

                self.add_chunk(
                    EmailDetails._replace(
                        receiver_email=email,
                        receiver_name=receiver_name,
                        receiver_lang=language,
                        subject=subject,
                        author_name=e.author.staff.inflections.subjective,
                        reply_to=e.author.staff.get_email(),
                    ),
                    render_to_string('not_owner.html', params),
                )
        return self.reply
