
import logging
from datetime import timedelta

from django.conf import settings as glob_settings
from django.core.exceptions import ObjectDoesNotExist
from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext as _

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, remove_watch, supply_events
from wiki.notifications.generators.utils import format_for_nice_datetime
from wiki.pages.logic.urls import url_to_wiki_page_in_frontend
from wiki.pages.logic import subscription
from wiki.pages.models import Revision
from wiki.pages.templatetags.evil_tags import declination_verb
from wiki.utils import timezone
from wiki.utils.backports.mds_compat import APIError
from wiki.utils.marked_diff import get_marked_diff_text

logger = logging.getLogger('generator.page.edition')

IS_INTRANET = getattr(glob_settings, 'IS_INTRANET', False)

# Максимальное количество дней от момента создания ревизии, когда она считается свежей и пригодной для уведомления
# пользователя об изменениях в ней.
MAX_REVISION_DAYS_COUNT = 30  # в днях

# Максимальный временной промежуток между ревизиями, изменения в которых могут быть склеяны в один diff.
MAX_REVISION_TIMEOUT = 24 * 60  # в минутах


def get_last_notified_revision(page, changes_started):
    """
    Получить последнюю ревизию, по которой было уведомление.
    У нас нет прямой связи между событием и ревизией, ориентируемся
    только на время их создания (на самом деле revision_id есть в meta,
    но неясно, насколько можно на него полагаться).
    Ревизия создается чуть раньше события (обычно сотые секунды),
    чтобы наверняка отфильтровать ревизию для начального события
    текущей нотификации, сделаем смещение на минуту
    """
    order = '-created_at'
    f_kw = {'page': page, 'created_at__lt': changes_started - timedelta(minutes=1)}
    return Revision.objects.filter(**f_kw).order_by(order).first()


def get_revisions_for_notification(page, changes_started):
    """
    Получить список ревизий, которые были после последнего уведомления.
    К этому списку прибавляется еще одна ревизия (если она существует): последняя ревизия,
    которая была раньше последнего уведомления или даты подписки. От нее мы будем отталкиваться
    при получении отличий между ревизиями.
    """
    f_kw = {'page': page}
    order = '-created_at'
    last_notif_rev = get_last_notified_revision(page, changes_started)
    if last_notif_rev:
        f_kw['created_at__gte'] = last_notif_rev.created_at
    revisions_after_last_notif = Revision.objects.filter(**f_kw).order_by(order)
    return revisions_after_last_notif


def get_revisions_pair_for_diff(revisions_for_notif, user):
    """
    Получить список пар ревизий (нач_ревизия, конеч_ревизия) для последующего анализа изменений между ними.
    """
    rev_pair = list()
    if len(revisions_for_notif) < 2:
        return rev_pair

    start_rev, end_rev = None, None
    for rev in revisions_for_notif:
        if not start_rev:
            start_rev = rev
            continue

        if rev.author == user:
            if end_rev:
                rev_pair.append([start_rev, end_rev])
                start_rev, end_rev = rev, None
            else:
                start_rev = rev
        else:
            end_rev = rev

    if start_rev and end_rev:
        rev_pair.append([start_rev, end_rev])
    return rev_pair


def remove_old_revisions(revisions_for_notif):
    """
    Удалить из списка ревизии старше определенного количества дней (MAX_REVISION_DAYS_COUNT).
    Удалить также те ревизии, которые должны быть склеяны с теми ревизиями, которые их поглощают, т.е. имеют одинакого
    автора и были созданы в течении относительно короткого промежутка времени (MAX_REVISION_TIMEOUT)
    """
    rev_list = list()
    for rev in revisions_for_notif:
        if (
            not rev_list
            or rev_list[0].author != rev.author
            or rev.id == revisions_for_notif.reverse()[0].id
            or rev_list[0].created_at - rev.created_at > timedelta(minutes=MAX_REVISION_TIMEOUT)
        ):
            rev_list.insert(0, rev)
        if len(rev_list) > 1 and (timezone.now() - rev.created_at).days > MAX_REVISION_DAYS_COUNT:
            break
    return rev_list


class EditionGen(BaseGen):
    """
    Generator for page edition events notifications.
    Regardless of edition events count, for every page subscriber will be generated exactly
    one notification with page body diff from his last viewed revision till current state.
    """

    _diff_cache = {}  # per-process cache for pages diff. Key - (start_rev.id, end_rev.id), value - diff text

    @supply_events(EventTypes.edit)
    def generate(self, events, generator_settings):
        if not events:
            return self.reply

        e = events[0]
        page = e.page

        # cause events ordered by created_at DESC
        changes_started = events[-1].created_at

        watchers = subscription.get_united_subscribed_users(page)

        for user in watchers:

            authors = self.get_authors(events, [user.username])
            if len(authors) == 0:
                # У событий нет других авторов, кроме текущего подписчика
                continue

            if glob_settings.IS_BUSINESS and user.is_dir_robot:
                remove_watch(user, page)
                continue

            try:
                if getattr(user.staff, 'is_dismissed', False):
                    continue
            except ObjectDoesNotExist:
                logger.warning(
                    "Can't generate a notification to the user '%s', because it is not found "
                    "in the table 'intranet_stuff'" % user.username
                )
                continue

            if not page.has_access(user):
                remove_watch(user, page)
                continue

            params = self.default_params.copy()
            e_lang = self.email_language(user, strict_mode=True)
            if e_lang is None:
                continue

            email, receiver_name, language = e_lang
            translation.activate(language)
            subject = _('Page edited: %(address)s titled %(title)s') % {
                'title': self.page_title_for_print(page),
                'address': self.page_name_for_print(page),
            }

            receiver_info = EmailDetails._replace(
                receiver_email=email, receiver_name=receiver_name, receiver_lang=language, subject=subject
            )
            if len(authors) == 1:
                for author in authors:
                    try:
                        receiver_info = receiver_info._replace(
                            author_name=author.staff.inflections.subjective, reply_to=author.staff.get_email()
                        )
                        params['author'] = author
                    except ObjectDoesNotExist:
                        logger.warning(
                            "Can't generate a notification to the user '%s', because the author '%s' "
                            "is not found in the table 'intranet_stuff'" % (user.username, author)
                        )
                        continue

            suffix_for_editing = ''
            if language == 'ru':
                if len(authors) == 1:
                    for author in authors:
                        suffix_for_editing = declination_verb(author.staff)
                else:
                    suffix_for_editing = 'и'

            params.update(
                {
                    'authors': authors,
                    'events': events,
                    'page_title': self.page_title_for_print(page),
                    'page': page,
                    'unsubscribe_url': url_to_wiki_page_in_frontend(page, action='unwatch'),
                    'suffix': suffix_for_editing,
                    'page_is_grid': False,
                    'legend_added_text': self.replace_tags_by_style(_('pages.Diff_AddedTextIsHighlightedByGreen')),
                    'legend_deleted_text': self.replace_tags_by_style(_('pages.Diff_DeletedTextIsHighlightedByRed')),
                }
            )

            revisions_for_notif = get_revisions_for_notification(page, changes_started)
            if not revisions_for_notif:
                continue

            revisions_for_notif = remove_old_revisions(revisions_for_notif)

            rev_pair = get_revisions_pair_for_diff(revisions_for_notif, user)
            for start_revision, end_revision in rev_pair:
                diff = self.get_page_diff(page, start_revision, end_revision)

                if diff is None:
                    # no changes in page for this subscriber
                    logger.warning(
                        'No changes in page %s for subscriber %s between revisions %s and %s'
                        % (page.supertag, user, start_revision.id, end_revision.id)
                    )
                    continue

                show_seconds = end_revision.created_at.minute == start_revision.created_at.minute

                params_copy = params.copy()
                params_copy.update(
                    {
                        'start_revision_time': start_revision.created_at,
                        'end_revision_time': end_revision.created_at,
                        'format_for_nice_datetime': format_for_nice_datetime(show_seconds),
                        'editions': diff,
                    }
                )

                self.add_chunk(receiver_info, render_to_string('edition.html', params_copy))

        return self.reply

    def get_authors(self, events, exclude_logins):
        authors = set([])
        for event in events:
            if event.author.username in exclude_logins:
                continue
            authors.add(event.author)
        return authors

    def replace_tags_by_style(self, text):
        return (
            text.replace('<del>', '<span style="background: #FAA;">')
            .replace('<ins>', '<span style="background: #AFA;">')
            .replace('</del>', '</span>')
            .replace('</ins>', '</span>')
        )

    def _get_cached_diff(self, start_rev, end_rev):
        """
        Get page diff text from cache or raise KeyError
        @type start_rev: Revision
        @type end_rev: Revision

        @rtype: list
        @return: diff
        @raise: KeyError
        """

        end_rev_id = None
        if end_rev:
            end_rev_id = end_rev.id

        return self._diff_cache[(start_rev.id, end_rev_id)]

    def _set_cached_diff(self, diff, start_rev, end_rev):
        """
        Set page diff text to cache
        @type diff: iterable
        @param diff: lines diff list
        @type start_rev: Revision
        @type end_rev: Revision
        """
        end_rev_id = None
        if end_rev:
            end_rev_id = end_rev.id

        self._diff_cache[(start_rev.id, end_rev_id)] = diff

    def _generate_diff(self, page, start_rev, end_rev):
        """
        Generate page text diff between {start_rev} and {end_rev} revisions
        @type page: Page
        @type start_rev: Revision
        @type end_rev: Revision

        @rtype: list
        @return: diff
        """
        non_empty_diff = False  # флаг пустого диффа

        try:
            new_text = end_rev.body
            old_text = start_rev.body
        except APIError:
            if not glob_settings.DEBUG:
                raise
            new_text = None
            old_text = None

        if new_text is None or old_text is None:
            diff = ({'txt': '<can not get blob>', 'class': None}, {'txt': '<can not get blob>', 'class': None})
        else:
            logger.info(
                'generating page {0} diff for {1}'.format(
                    page.supertag, (start_rev.id, end_rev.id if end_rev else None)
                )
            )
            diff = get_marked_diff_text(old_text, new_text, skip_unchanged_lines=True)

        for line in diff:
            if line:
                non_empty_diff = True
                line['txt'] = self.replace_tags_by_style(line['txt'])

        diff = diff if non_empty_diff else None  # replace empty diff with None
        self._set_cached_diff(diff, start_rev, end_rev)  # send diff to cache

        return diff

    def get_page_diff(self, page, start_rev, end_rev):
        """
        Return body text diff for page {page} between {start_rev} and {end_rev} revisions

        @type page: Page
        @type start_rev: Revision
        @type end_rev: Revision

        @rtype: list
        @return: diff
        """
        try:
            diff = self._get_cached_diff(start_rev, end_rev)
            logger.info(
                'got diff from cache for page {0} revs {1}'.format(
                    page.supertag, (start_rev.id, end_rev.id if end_rev else None)
                )
            )
        except KeyError:
            logger.info(
                'diff cache miss for page {0} revs {1}'.format(
                    page.supertag, (start_rev.id, end_rev.id if end_rev else None)
                )
            )
            # no diff in cache yet
            diff = self._generate_diff(page, start_rev, end_rev)

        return diff
