import logging

from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext
from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, supply_events
from wiki.pages.logic.authors_add_chief import add_first_found_chief_to_authors
from wiki.pages.logic.urls import url_to_wiki_page_in_frontend
from wiki.pages.models import AccessRequest

logger = logging.getLogger(__name__)


class AccessRequestGen(BaseGen):
    @supply_events(EventTypes.request_access)
    def generate(self, events, settings):
        if events:
            page = events[0].page
            for event in events:
                meta = event.meta
                try:
                    areq = AccessRequest.objects.get(pk=meta['access_request_id'])
                except AccessRequest.DoesNotExist:
                    logger.exception(
                        'Somehow I could not get AccessRequest:id=%s, event: %s' % (meta['access_request_id'], event.pk)
                    )
                    continue

                title = page.title
                grant = url_to_wiki_page_in_frontend(
                    page=page, query_args={'action': 'allow', 'id': areq.pk}, action='requestaccess'
                )
                deny = url_to_wiki_page_in_frontend(
                    page=page, query_args={'action': 'deny', 'id': areq.pk}, action='requestaccess'
                )
                params = self.default_params.copy()
                params.update(
                    {
                        'grant': grant,
                        'deny': deny,
                        'reason': areq.reason,
                        'title': self.page_title_for_print(page),
                        'page_title': title,
                        'event': event,
                        'page_tag': event.page.tag,
                    }
                )

                try:
                    if page.is_all_authors_dismissed():
                        add_first_found_chief_to_authors(page)
                except Exception:
                    logger.exception(f'Can not add author as chief to page "{page.supertag}"')

                for email, receiver_name, language in self.get_page_authors_emails(page):
                    translation.activate(language)

                    subject = ugettext('Access request: to page %(name)s titled %(title)s') % {
                        'title': self.page_title_for_print(page),
                        'name': self.page_name_for_print(page),
                    }

                    self.add_chunk(
                        EmailDetails._replace(
                            receiver_email=email,
                            receiver_name=receiver_name,
                            receiver_lang=language,
                            subject=subject,
                            author_name=event.author.staff.inflections.subjective,
                            reply_to=event.author.staff.get_email(),
                        ),
                        render_to_string('request_access.html', params),
                    )

        return self.reply
