
from django.conf import settings as glob_settings
from django.template.loader import render_to_string
from django.utils import translation

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, supply_events
from wiki.users.core import get_support_contact


class RequestOwnershipGen(BaseGen):
    @supply_events(EventTypes.request_ownership)
    def generate(self, events, settings):
        for event in events:
            email_subject = 'Запрос на изменение владельца страниц пользователя {0}@'.format(
                event.page.owner and event.page.owner.username
            )

            is_owner_dismissed = event.page.owner and getattr(event.page.owner.staff, 'is_dismissed', False)
            if is_owner_dismissed or (event.page.owner is None):
                if glob_settings.IS_BUSINESS:
                    from wiki.users_biz.dao import get_admin_group_members

                    admin_users = get_admin_group_members()
                    receiver_emails = [admin.email for admin in admin_users]
                    if len(receiver_emails) == 1:
                        receiver_email, receiver_name, receiver_lang = self.email_language(admin_users[0])
                    else:
                        receiver_email, receiver_name, receiver_lang = tuple(receiver_emails), '', 'ru'
                else:
                    receiver_email, receiver_name, receiver_lang = (
                        glob_settings.SUPPORT_EMAIL,
                        get_support_contact(),
                        'ru',
                    )
            else:
                receiver_email, receiver_name, receiver_lang = self.email_language(event.page.owner)

            translation.activate(receiver_lang)

            email_details = EmailDetails._replace(
                receiver_email=receiver_email,
                receiver_name=receiver_name,
                receiver_lang=receiver_lang,
                subject=email_subject,
                author_name=event.author.username,
                cc=event.author.email,
            )

            meta = event.meta

            message = render_to_string(
                'notifications/request_ownership.html',
                {
                    'requester': event.author.username,
                    'cluster': event.page.absolute_url,
                    'owner': event.page.owner and event.page.owner.username,
                    'is_dismissed': is_owner_dismissed,
                    'root_page_only': meta.get('root_page_only', False),
                    'reason': meta['reason'],
                },
            )
            self.add_chunk(email_details, message)
            return self.reply
