from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, supply_events
from wiki.pages.logic import url_to_wiki_page_in_frontend


class RequestPageAuthorGen(BaseGen):
    @supply_events(EventTypes.request_page_author)
    def generate(self, events, settings):
        for event in events:
            approve = url_to_wiki_page_in_frontend(
                page=event.page, query_args={'new_author': event.author.username}, action='request_page_author'
            )

            for email, receiver_name, language in self.get_page_authors_emails(event.page):

                translation.activate(language)

                # Translators:
                #   ru: Запрос в авторы: %(username)s хочет стать автором страницы %(page)s
                #   en: Authorship request: %(username)s wants to be the author of the page %(page)s
                email_subject = ugettext(
                    'Authorship request: %(username)s wants to be the author of the page %(page)s'
                ) % {'username': event.author.username, 'page': self.page_name_for_print(event.page)}

                email_details = EmailDetails._replace(
                    receiver_email=email,
                    receiver_name=receiver_name,
                    receiver_lang=language,
                    subject=email_subject,
                    author_name=event.author.username,
                    cc=event.author.email,
                )

                meta = event.meta

                message = render_to_string(
                    'request_page_author.html',
                    {
                        'reason': meta['reason'],
                        'event': event,
                        'approve': approve,
                        'title': self.page_title_for_print(event.page),
                        'page_tag': event.page.tag,
                    },
                )
                self.add_chunk(email_details, message)

        return self.reply
