
import logging

from django.conf import settings as glob_settings
from django.contrib.auth import get_user_model
from django.template.loader import render_to_string
from django.utils import translation
from django.utils.translation import ugettext

from wiki.notifications.generators.base import BaseGen, EmailDetails, EventTypes, remove_watch, supply_events
from wiki.org import get_org, org_user
from wiki.pages.models import Page

IS_INTRANET = getattr(glob_settings, 'IS_INTRANET', False)

logger = logging.getLogger()


class WatchGen(BaseGen):
    @supply_events([EventTypes.watch, EventTypes.subscribe_other_user])
    def generate(self, events, settings=None):
        self.cluster_subscriptions(events)
        self.page_subscriptions(events)
        return self.reply

    def page_subscriptions(self, events):
        events = [e for e in events if not e.meta.get('is_cluster', False)]
        if not events:
            return
        e = events[0]
        # cause events ordered by created_at DESC
        for author in e.page.get_authors():
            if author == e.author:
                continue

            user = author
            if getattr(user.staff, 'is_dismissed', False):
                continue
            if not e.page.has_access(user):
                remove_watch(user, e.page)
                continue
            tmp = self.email_language(user, strict_mode=True)
            if tmp is None:
                continue
            email, name, lang = tmp
            translation.activate(lang)
            gender = 'M'
            params = self.default_params.copy()
            if e.author.staff and getattr(e.author.staff, 'gender', False):
                gender = e.author.staff.gender
            params['subscription_suffix'] = ''
            if translation.get_language()[:2] == 'ru':
                if gender == 'F':
                    params['subscription_suffix'] = 'ась'
                else:
                    params['subscription_suffix'] = 'ся'

            subject = ugettext('Subscription: to page %(page_tag)s titled %(page_title)s') % {
                'page_tag': self.page_name_for_print(e.page),
                'page_title': self.page_title_for_print(e.page),
            }

            params.update(
                {
                    'event': e,
                    'page_title': self.page_title_for_print(e.page),
                }
            )

            self.add_chunk(
                EmailDetails._replace(
                    receiver_email=email,
                    receiver_name=name,
                    receiver_lang=lang,
                    subject=subject,
                    author_name=e.author.staff.inflections.subjective,
                ),
                render_to_string('watch.html', params),
            )

    def cluster_subscriptions(self, events):
        """
        get pages of this owner
        learn subscribers which subscribed to theese pages
        for each owner and each subscriber generate the letter of all owner pages
        """
        cluster_events = [e for e in events if e.meta.get('is_cluster')]
        if not cluster_events:
            return
        empty_list = []
        avoid_doubles = []
        user_model = get_user_model()
        for e in cluster_events:
            subscribed_user_id = e.meta.get('subscribed_user_id')
            if subscribed_user_id:
                try:
                    subscribed_user = org_user().get(id=subscribed_user_id)
                except user_model.DoesNotExist:
                    logger.error("Can't get user with id='%s'" % subscribed_user_id)
                    continue
            else:
                subscribed_user = e.author

            tmp = e.meta.get('pages') or empty_list
            comment = e.meta.get('comment')
            pages = Page.objects.filter(supertag__in=tmp, org=get_org())
            authors = set()
            for page in pages:
                authors.update(page.get_authors())
            # authors of root page
            subscription_suffix = 'ся'
            if getattr(subscribed_user.staff, 'gender', 'M') == 'F':
                subscription_suffix = 'ась'
            params = {
                'person': subscribed_user,
                'page_title': self.page_title_for_print(e.page),
                'page': e.page,
                'event': e,
                'is_owner': False,
                'page_num': len(pages),
                'subscription_suffix': '',
            }

            for author in authors:
                if author in e.page.get_authors() or author == subscribed_user:
                    continue
                is_double = (author, subscribed_user, e.page)
                if is_double in avoid_doubles:
                    continue
                avoid_doubles.append(is_double)
                receiver = author
                ed = self.email_details_for(receiver)
                if ed is None:
                    # детали не пришли если
                    # - сотрудник уволен
                    # - адресат робот
                    # - нельзя найти почту
                    continue

                if ed.receiver_lang == 'ru':
                    params['subscription_suffix'] = subscription_suffix
                else:
                    params['subscription_suffix'] = ''
                ed = ed._replace(
                    subject=self.build_masswatch_subject(e.page),
                    author_name=subscribed_user.staff.inflections.subjective,
                )
                params['page_num'] = len([p for p in pages if author in p.get_authors()])
                self.add_chunk(ed, render_to_string('notifications/mass_watch.html', params))

            for author in e.page.get_authors():
                if author != subscribed_user:
                    is_double = (author, subscribed_user, e.page)
                    if is_double in avoid_doubles:
                        continue
                    avoid_doubles.append(is_double)
                    receiver = author
                    ed = self.email_details_for(receiver)
                    if ed is None:
                        # это ок ситуация - см выше
                        continue
                    ed = ed._replace(
                        subject=self.build_masswatch_subject(e.page),
                        author_name=subscribed_user.staff.inflections.subjective,
                    )
                    if ed.receiver_lang == 'ru':
                        params['subscription_suffix'] = subscription_suffix
                    else:
                        params['subscription_suffix'] = ''
                    params['is_owner'] = True
                    self.add_chunk(ed, render_to_string('notifications/mass_watch.html', params))

            # выслать уведомления пользователям, которых подписали на кластер
            if subscribed_user.id != e.author.id:
                is_double = (e.author, subscribed_user, e.page, comment)
                if is_double in avoid_doubles:
                    continue
                avoid_doubles.append(is_double)

                receiver = subscribed_user
                ed = self.email_details_for(receiver)

                if not ed:
                    # это ок ситуация - см выше
                    continue

                ed = ed._replace(
                    subject=self.build_masswatch_subject(e.page), author_name=e.author.staff.inflections.subjective
                )

                if ed.receiver_lang == 'ru' and getattr(e.author.staff, 'gender', 'M') == 'F':
                    params['subscription_suffix'] = 'а'
                else:
                    params['subscription_suffix'] = ''

                if comment:
                    params['comment'] = comment

                params['person'] = e.author

                self.add_chunk(ed, render_to_string('notifications/mass_watch_other_users.html', params))

    def build_masswatch_subject(self, page):
        return ugettext('notifications.masswatch:SubscriptionSubject %(page_tag)s %(page_title)s') % {
            'page_tag': self.page_name_for_print(page),
            'page_title': self.page_title_for_print(page),
        }

    def email_details_for(self, user):
        tmp = self.email_language(user, strict_mode=True)
        if tmp is None:
            return None
        email, name, lang = tmp
        translation.activate(lang)
        return EmailDetails._replace(receiver_email=email, receiver_name=name, receiver_lang=lang)
