
import base64
from datetime import timedelta

from wiki.notifications.generators.base import DEFAULT_EVENTS_PARAMS
from wiki.notifications.models import PageEvent
from wiki.utils import timezone


def create_page_event(page, user, event_type, meta=None, **event_params):
    if isinstance(event_type, str):
        event_type = getattr(PageEvent.EVENT_TYPES, event_type)

    params = DEFAULT_EVENTS_PARAMS[('page', event_type)].copy()
    timeout = params.get('timeout', 0)
    params['timeout'] = timezone.now() + timedelta(minutes=timeout)
    params.update(event_params)

    event = PageEvent(page=page, event_type=event_type, author=user, **params)
    if meta is None:
        meta = {}
    event.meta = meta
    event.save()
    return event


def create_page_event_create(page, user, revision, **event_params):
    meta = {'revision_id': revision.id}
    return create_page_event(page=page, user=user, event_type='create', meta=meta, **event_params)


def create_page_event_edit(page, user, revision, **event_params):
    from wiki.utils.diff import simple_diff

    added, deleted = simple_diff(page._old_body, page.body)
    meta = {
        'added': '\n'.join(added),
        'deleted': '\n'.join(deleted),
        'revision_id': revision.id,
    }
    return create_page_event(
        page=page, user=user, event_type='edit', meta=meta, created_at=page.modified_at, **event_params
    )


def create_page_event_keywords(page, user, **event_params):
    keywords_string = str(page.keywords).encode('UTF-8')
    meta = {'keywords': base64.encodebytes(keywords_string).decode('UTF-8')}
    return create_page_event(page=page, user=user, event_type='keywords', meta=meta, **event_params)


def create_page_event_change_to_redirect(page, user, destination, **event_params):
    meta = {'redirect_to': destination.supertag}
    if not page.has_redirect():
        meta['page_type'] = page.page_type
    return create_page_event(page=page, user=user, event_type='change_to_redirect', meta=meta, **event_params)


def create_page_event_revert_from_redirect(page, user, **event_params):
    return create_page_event(page=page, user=user, event_type='revert_from_redirect', meta={}, **event_params)


def create_page_event_add_comment(page, user, comment, **event_params):
    meta = {'comment_id': comment.id}
    return create_page_event(page=page, user=user, event_type='add_comment', meta=meta, **event_params)
