
import logging

from django.core.management.base import BaseCommand

from wiki.middleware.read_only import service_is_readonly
from wiki.notifications.queue import Queue
from wiki.notifications.tasks import NotifyPageSubscribersTask

logger = logging.getLogger('wiki.management_commands')


class Command(BaseCommand):
    help = 'Notify users of page changes synchronously'
    requires_system_checks = False

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument(
            '--simulate',
            '-s',
            action='store_true',
            dest='simulate',
            default=False,
            help='Only generate letters without sending',
        )
        parser.add_argument(
            '--keep',
            '-k',
            action='store_true',
            dest='keepletters',
            default=False,
            help='Don\'t delete messages from queue',
        )

    def handle(self, *args, **options):
        try:
            self.do_handle(*args, **options)
        except Exception as e:
            logger.exception('Unhandled error at notify command "%s"', repr(e))

    def do_handle(self, *args, **options):
        """
        Синхронно разослать письма.

        @param kwargs: simulate - не отсылать письма, только генерировать
        @param kwargs: sync - отсылать письма синхронно
        @param kwargs: keepletters - не помечать события обработанными.
        При следующем запуске они снова отправятся.
        """
        self.verbosity = int(options.get('verbosity'))

        if not service_is_readonly():
            simulate = options.get('simulate')  # only generate mail, no sending
            if self.verbosity > 0 and simulate:
                logger.info('Simulation mode - no real sending!')

            pages_with_events = Queue.page_ids_with_events()
            sync = options.get('sync')

            for page_id in pages_with_events:
                logger.info('processing page "%s"', page_id)
                task = NotifyPageSubscribersTask()
                if sync:
                    task(page_id=page_id, options=options)
                else:
                    task.delay(page_id=page_id, options=options)
