from random import randint

from django.conf import settings
from django.db import models
from django.utils.timezone import now

from wiki.notifications.generators.base import EventTypes
from wiki.pages.models import Page
from wiki.utils.mds_json_field import MdsJsonField
from wiki.utils.storage import STORAGE


def make_pageevent_storage_id():
    """
    Возвращает уникальную строку storage_id для PageEvent.
    """
    return '{code}:pageevent:{datetime_now}:{random_int}'.format(
        code=settings.WIKI_CODE,
        datetime_now=now().strftime('%Y-%m-%d %H:%M:%S:%f'),
        random_int=str(randint(10**5, 10**6 - 1)),
    )


class PageEventMetaField(MdsJsonField):
    def make_storage_id(self):
        return make_pageevent_storage_id()


class PageEvent(models.Model):
    objects: models.Manager

    class Meta:
        db_table = 'notifications_pageevent'

    EVENT_TYPES = EventTypes
    created_at = models.DateTimeField(auto_now_add=True)
    timeout = models.DateTimeField(null=False)
    page = models.ForeignKey(Page, on_delete=models.CASCADE)
    author = models.ForeignKey(settings.AUTH_USER_MODEL, related_name='author', on_delete=models.CASCADE)
    sent_at = models.DateTimeField(null=True, db_index=True)
    event_type = models.IntegerField(
        choices=list(EventTypes._asdict().items()), default=EventTypes.create, blank=False, db_index=True
    )
    notify = models.BooleanField(default=True)

    meta = PageEventMetaField(storage=STORAGE, null=False)

    def __str__(self):
        etype = [e for e in list(EventTypes._asdict().items()) if e[1] == self.event_type]
        if etype:
            etype = etype[0][0]
        else:
            etype = 'UNKNOWN'
        try:
            return '%s in "%s" (%s)' % (etype, self.page.supertag, self.pk)
        except Page.DoesNotExist:
            return '%s in %s (%s)' % (etype, 'unknown page', self.pk)
