
import logging

from django_replicated.utils import routers

from wiki.notifications.models import PageEvent
from wiki.pages.models import Page
from wiki.utils import timezone

logger = logging.getLogger(__name__)
from django.db import connection


class Queue(object):
    @classmethod
    def remove(self, events):
        """
        Быстро пометить события обработанными
        """
        PageEvent.objects.filter(id__in=[e.pk for e in events]).update(sent_at=timezone.now())

    @classmethod
    def page_ids_with_events(self):
        """
        Return page ids set for pages with unsent events.

        @rtype: set
        """

        # Это тоже не самое оптимальное, но хотя бы это не полторы минуты, а 7 секунд отработает
        # По-хорошему надо все менять архитектурно

        with connection.cursor() as cursor:
            cursor.execute(
                """SELECT DISTINCT pages_page.id FROM notifications_pageevent
                        JOIN pages_page ON
                            notifications_pageevent.page_id = pages_page.id

WHERE notifications_pageevent.notify = TRUE
        AND notifications_pageevent.sent_at IS NULL
        AND notifications_pageevent.created_at > NOW() - INTERVAL '7 days'
        AND (pages_page.status != 0 OR notifications_pageevent.event_type = %s); """,
                [PageEvent.EVENT_TYPES.delete],
            )

            rows = cursor.fetchall()
        page_ids = set([p[0] for p in rows])
        return page_ids

    def new_events(self, page_id):
        """
        Отфильтровать пары (страница, событие), по которым еще не наступил таймаут
        @type page_id: int
        @rtype: QuerySet

        TODO: здесь должен быть @classmethod

        Не дает уведомить о таких событиях, которые "продолжают происходить",
        например, страница часто редактируется, а скрипт уведомления работает
        изредка, тогда уведомление о редактировании не поступит.

        Такие пары не должны попасть в письмо-уведомление
        """
        try:
            routers.use_state('slave')
            return self._new_events(page_id)
        except Exception as e:
            logger.exception(f'Exception while getting new events {e}')
        finally:
            routers.revert()

    def _new_events(self, page_id):
        page = Page.objects.get(id=page_id)
        is_deleted = page.status == 0

        qfilter = {'sent_at': None, 'notify': True, 'page': page_id}

        if is_deleted:
            qfilter['event_type'] = PageEvent.EVENT_TYPES.delete

        # легкий запрос возвращает все события по которым надо уведомить
        events_candidates = PageEvent.objects.filter(**qfilter).values_list('event_type', 'page_id', 'id', 'timeout')

        NOW = timezone.now()

        omit_tuples = set(
            (event[0], event[1]) for event in events_candidates if event[3] > NOW  # event_type, page_id
        )  # timeout не наступил
        logger.debug('Omit those event tuples %s', omit_tuples)

        old_events_ids = dict(
            (
                (event[2], (event[0], event[1]))  # event_type, page_id
                for event in events_candidates
                if event[3] <= NOW  # таймаут наступил
            )
        )

        # пропускаем те, по которым есть еще не наступившие таймауты
        event_ids_to_notify = (id for id in old_events_ids if old_events_ids[id] not in omit_tuples)
        return (
            PageEvent.objects.filter(id__in=event_ids_to_notify)
            .select_related('page', 'author')
            .order_by('page', 'event_type', '-created_at')
        )
