
import datetime
import logging

import django.dispatch
from django.conf import settings

from wiki.notifications.generators import base
from wiki.notifications.models import PageEvent
from wiki.utils import timezone

log = logging.getLogger(__name__)

access_revoked = django.dispatch.Signal(providing_args=['access', 'author'])
access_granted = django.dispatch.Signal(providing_args=['access', 'author'])
keywords_changed = django.dispatch.Signal(providing_args=['page', 'keywords'])
page_access_settings_changed = django.dispatch.Signal(
    providing_args=[
        'page',
        'author_of_changes',
        'old_access_type',
        'new_access_type',
        'added_users',
        'added_groups',
        'removed_users',
        'removed_groups',
    ]
)


@django.dispatch.receiver(access_revoked, weak=False)
def notify_access_revoked(sender, access, author, **kwargs):
    meta = {
        'staff_id': access.staff_id,
        'is_common': access.is_common,
        'is_owner': access.is_owner,
        'resolution_type': 'revoked',
        'group_id': access.group_id,
    }
    PageEvent(
        page=access.page,
        event_type=base.EventTypes.resolve_access,
        author=author,
        timeout=timezone.now() + datetime.timedelta(minutes=10),
        notify=False,
        meta=meta,
    ).save()


@django.dispatch.receiver(access_granted, weak=False)
def notify_access_granted(sender, access, author, **kwargs):
    meta = {
        'staff_id': access.staff_id,
        'is_common': access.is_common,
        'is_owner': access.is_owner,
        'resolution_type': 'granted',
        'group_id': access.group_id,
    }

    if 'notify_all' in kwargs and kwargs['notify_all']:
        meta['notify_all'] = True
    else:
        meta['notify_all'] = False

    if 'access_request' in kwargs:
        meta['access_request_id'] = kwargs['access_request'].pk

    set_to_inherit = kwargs.get('inherited', False)
    if set_to_inherit:
        meta['inherited'] = set_to_inherit

    PageEvent(
        page=access.page,
        event_type=base.EventTypes.resolve_access,
        author=author,
        timeout=timezone.now() + datetime.timedelta(minutes=10),
        notify=True if 'notify' not in kwargs else kwargs['notify'],
        meta=meta,
    ).save()


@django.dispatch.receiver(page_access_settings_changed)
def notify_page_access_settings_changed(
    sender,
    page,
    author_of_changes,
    old_access_type,
    new_access_type,
    added_users,
    added_groups,
    removed_users,
    removed_groups,
    **kwargs
):
    meta = {
        'old_access_type': old_access_type,
        'new_access_type': new_access_type,
    }
    if added_users:
        meta['added_users'] = [staff.username for staff in added_users]

    if removed_users:
        meta['removed_users'] = [staff.username for staff in removed_users]

    if added_groups:
        meta['added_groups'] = [group.group.title if settings.IS_BUSINESS else group.name for group in added_groups]

    if removed_groups:
        meta['removed_groups'] = [group.group.title if settings.IS_BUSINESS else group.name for group in removed_groups]

    PageEvent(
        page=page,
        author=author_of_changes,
        timeout=timezone.now() + datetime.timedelta(minutes=5),
        event_type=base.EventTypes.change_access,
        notify=True,
        meta=meta,
    ).save()


def request_access_created(instance, **kwargs):
    """
    Listen to AccessRequest created/updated
    """
    timeout = timezone.now() + datetime.timedelta(minutes=1)
    PageEvent(
        page=instance.page,
        event_type=base.EventTypes.request_access,
        author=instance.applicant,
        meta={'reason': instance.reason, 'access_request_id': instance.pk},
        timeout=timeout,
        notify=True,
    ).save()


def request_ownership_created(page, requester, root_page_only, reason):
    PageEvent(
        page=page,
        event_type=base.EventTypes.request_ownership,
        author=requester,
        meta={
            'root_page_only': root_page_only,
            'reason': reason,
        },
        timeout=timezone.now() + datetime.timedelta(minutes=1),
        notify=True,
    ).save()


def request_page_author_created(page, requester, reason):
    PageEvent(
        page=page,
        event_type=base.EventTypes.request_page_author,
        author=requester,
        meta={
            'reason': reason,
        },
        timeout=timezone.now() + datetime.timedelta(minutes=1),
        notify=True,
    ).save()
