import logging
import smtplib
import socket

from celery.exceptions import SoftTimeLimitExceeded
from django.conf import settings
from django.utils import translation
from django.utils.translation import ugettext
from ylog.context import log_context

from wiki.notifications import generators
from wiki.notifications.generators import base
from wiki.notifications.queue import Queue
from wiki.org import org_ctx
from wiki.pages.models import Page
from wiki.utils.mail import build_wiki_email_message
from wiki.utils.tasks.base import LockedCallableTask
from wiki.utils.backports.smtp_backport import BackportMonkeyPatch

logger = logging.getLogger('wiki.tasks')
console = logging.getLogger('wiki.management_commands')


class NotifyPageSubscribersTask(LockedCallableTask):
    name = 'wiki.notify_page_subscribers'
    logger = logger
    time_limit = 60 * 10
    lock_name_tpl = 'notify_page_{page_id}_subscripers'  # шаблон имени уникального для page_id таска

    def run(self, *args, **kwargs):
        try:
            return notify_page_subscribers(*args, **kwargs)
        except SoftTimeLimitExceeded:
            # таск выполняется слишком долго, скоро его воркер будет убит celery.
            # таска может быть перезапущена самой селери,
            # а может быть создана идентичная при следующем запуске команды notify.
            self.logger.warning(
                'Notifications for page "%s" were sent partially because of timeout. Rerun it somehow',
                kwargs.get('page_id'),
            )
            raise


def send_event_email_message(page, event_type, details, chunks, simulate=False):
    message = build_event_email_message(page=page, event_type=event_type, details=details, chunks=chunks)
    if simulate:
        return True

    attempts = 3
    while attempts >= 0:
        attempts -= 1
        try:
            message.send()
            return True
        except smtplib.SMTPRecipientsRefused:
            # На Wiki ШАД не у каждого пользователя есть email (?!)
            # и генератор писем может вернуть что-нибудь вроде
            # "ФИО <>" вместо "ФИО <me@example.com>"
            # TODO: Выяснить откуда берутся пользователи без email-ов
            logger.warning(
                'Failed to send email:to="%s", type=%s',
                details.receiver_email,
                event_type,
            )
            return False

        except (smtplib.SMTPServerDisconnected, smtplib.SMTPConnectError):
            logger.warning('SMTP connection error while sending the email, will retry')
            if hasattr(message, 'connection'):
                # иначе джанго будет пытаться отправлять через мёртвый конекнш и будет вылетать
                # SMTPServerDisconnected: please run connect() first
                message.connection = None
        except socket.error:
            logger.warning('Socket error while sending the email, will retry')

    logger.exception('Send event email: out of attempts')
    return False


def notify_page_subscribers(page_id, options=None):
    """
    Notify subscribers about new events for specific page
    @type page_id: int
    @param page_id: page id
    @param options: options dict-like object
    """
    count = 0
    skipped = 0

    # get new events for page
    events = list(Queue().new_events(page_id))

    if not events:
        return

    options = options or {}
    simulate = options.get('simulate', False)

    page = Page.objects.get(pk=page_id)
    handler = page.page_type

    with BackportMonkeyPatch(), log_context(slug=page.slug):
        logger.info('generating notifications for page "%s" for %s events', page.supertag, len(events))

        if handler not in (Page.TYPES.PAGE, Page.TYPES.GRID):
            logger.info(
                'Page %s has events but is not page or grid. It is %s. Events: %s'
                % (page.supertag, str(handler), str(events))
            )
            handler = Page.TYPES.PAGE

        for generator in generators.ORDER[handler]:
            event_type = str(generator.__class__.__name__).lower()
            if event_type.endswith('gen'):
                event_type = event_type[:-3]
            letters = None

            try:
                generator_settings = base.SETTINGS.copy()
                if simulate:
                    generator_settings['simulate'] = True
                with org_ctx(page.org):
                    letters = generator.generate(events, generator_settings)
            except Exception:
                logger.exception('Notify generator "%s" failed', generator.__class__)
                # в этом случае отсылаем все что удалось накопить.
                # Остальные, несозданные письма, помечаем успешно отправленными.
                # Это не совсем верно, но не критично.

            letters = letters or {}

            for to, chunks in letters.items():
                logger.info(
                    'Sending email:to="%s", event_type="%s", page "%s"', to.receiver_email, event_type, page.supertag
                )

                try:
                    if send_event_email_message(
                        page=page, event_type=event_type, details=to, chunks=chunks, simulate=simulate
                    ):
                        count += 1
                        continue
                except Exception:
                    logger.exception('Sending failed')

                skipped += 1

        if not (options.get('keepletters', False) or simulate):
            Queue.remove(events)

        logger.info(
            'Page {tag} and {count_events} events resulted in {count} emails sent and {skipped} emails skipped'.format(
                count=count, skipped=skipped, tag=page.supertag, count_events=len(events)
            )
        )


def build_event_email_message(page, event_type, details, chunks):
    """
    Build email message notifying subscriber about page event

    @type page: Page model
    @type event_type: basestring
    @type details: wiki.notifications.generators.base.EmailDetails
    @type chunks: iterable

    @rtype: EmailMessage
    @return: composed message object
    """
    translation.activate(details.receiver_lang)

    subject = details.subject or ugettext('Changes on wiki page "%s"') % (page.title or page.tag)
    author_name = details.author_name

    # WATCH - I add the last new line symbol, otherwise it gets stripped out SOMEHOW.
    body = address_person(details.receiver_name)
    body += notification_count_msg(len(chunks))
    body += '\n<br />'.join(chunks)
    body += '\n<br />' + how_to_manage_subscriptions()
    body += '\n<p>%s</p>\n' % signature(no_reply=details.reply_to is None)

    return build_wiki_email_message(
        to=details.receiver_email,
        cc=details.cc,
        bcc=details.bcc,
        reply_to=details.reply_to,
        subject=subject,
        body=body,
        author_name=author_name,
        content_subtype='html',
        supertag=page.supertag,
        headers={
            'X-Yandex-Wiki-Notification': event_type,
        },
    )


def notification_count_msg(count):
    msg_notif_count = ''
    if count > 1:
        # Translators:
        #   ru: Это письмо содержит несколько уведомлений ({notif_count}) о редактировании одной страницы.
        #   en: This letter contains several notifications ({notif_count}) about editing a single page.
        msg_notif_count = '\n<p>%s\n<p/>\n<br />' % ugettext('notifications:Tasks:NotifCountMsg {notif_count}').format(
            notif_count=count
        )
    return msg_notif_count


def address_person(name):
    if name:
        return '%s, %s!' % (ugettext('Hello'), name)
    else:
        return '%s!' % ugettext('Hello')


def how_to_manage_subscriptions():
    return '<a href="{url}">{text}</a>'.format(
        url=settings.MANAGE_SUBSCRIPTIONS_URL, text=ugettext('How to manage subscriptions')
    )


def signature(no_reply=False):
    result = ugettext('Your Wiki') + '.'
    if no_reply:
        result += "\n<br />\n<em><font size='-1'>{no_reply}.</font></em>\n".format(
            no_reply=ugettext('Do not reply to this message, please')
        )
    return result


def log_receivers(to, page, events, options=None):
    """
    Log notifying email receivers
    """
    options = options or {}
    verbosity = int(options.get('verbosity', 0))

    cc_str = '(Cc: %s)' % str(to.cc) if to.cc else ''
    if verbosity > 2:
        console.info(
            'Send email to "{0}" "{1}" on page "{2}" events: {3}'.format(
                to.receiver_email, cc_str, page.supertag, events
            )
        )
    elif verbosity > 1:
        console.info('Send email to "{0}" "{1}" on page "{2}"'.format(to.receiver_email, cc_str, page.supertag))
    elif verbosity > 0:
        console.info('Send email to "{0}" "{1}"'.format(to.receiver_email, cc_str))
