from django.conf import settings

from wiki.intranet.models import Staff
from wiki.notifications.models import PageEvent
from wiki.pages.logic import url_to_wiki_page_in_frontend
from wiki.pages.models import Page, AccessRequest

from wiki.notifications_v2.request_access.schemas import NotifyDataForRequestAccess, AccessLinkField
from wiki.notifications_v2.utils import (
    create_declination_verb,
    create_inflect_field,
    create_name_field,
    create_page_field,
    create_staff_field,
)


def create_access_links(page: Page, access_request_id: int) -> AccessLinkField:
    grant = url_to_wiki_page_in_frontend(
        page=page, query_args={'action': 'allow', 'id': access_request_id}, action='requestaccess'
    )
    deny = url_to_wiki_page_in_frontend(
        page=page, query_args={'action': 'deny', 'id': access_request_id}, action='requestaccess'
    )
    return AccessLinkField(grant=grant, deny=deny)


def create_message_data_for_request_access(
    page: Page, staff: Staff, reason: str, access_request_id: int
) -> NotifyDataForRequestAccess:
    user_name = create_name_field(staff)
    return NotifyDataForRequestAccess(
        authorName=None,  # заполняется после
        userStaff=create_staff_field(user_name, login=staff.login),
        userName=user_name,
        userInflect=create_inflect_field(staff, forms=['genitive', 'dative']),
        page=create_page_field(page),
        accessLink=create_access_links(page, access_request_id),
        userVerb=create_declination_verb(staff),
        reason=reason,
        docLink=settings.MANAGE_SUBSCRIPTIONS_URL,
    )


def create_messages(event: PageEvent) -> list[tuple[str, NotifyDataForRequestAccess]]:
    access_request = AccessRequest.objects.get(pk=event.meta['access_request_id'])
    template_message = create_message_data_for_request_access(
        page=event.page,
        staff=event.author.staff,
        reason=access_request.reason,
        access_request_id=access_request.id,
    )

    cloud_uid_and_messages = []
    for page_author in event.page.authors.all():
        message = template_message.copy()
        message.authorName = create_name_field(page_author.staff)

        cloud_uid_and_messages.append((page_author.get_cloud_uid(), message))

    return cloud_uid_and_messages
