from django.conf import settings
from django.utils import translation

from wiki.intranet.models import Staff
from wiki.intranet.models.consts import GENDER
from wiki.legacy.translit import name_translit
from wiki.notifications_v2.schemas import PageField, EnRuField, UserInflectField
from wiki.pages.models import Page


def shorten_slug(slug: str) -> str:
    if slug.count('/') > 2:
        spl = slug.split('/')
        slug = '/'.join(spl[:1] + ['...'] + spl[-2:])
    return '/' + slug


def create_page_field(page: Page) -> PageField:
    return PageField(
        slug=page.supertag_before_deletion,
        shortSlug=shorten_slug(page.supertag_before_deletion),
        title=page.title,
        link=page.absolute_url_before_deletion,
    )


def create_name_field(staff: Staff) -> EnRuField:
    login = staff.login.split('@')[0]

    first_name = staff.first_name
    last_name = staff.last_name
    name = _clear_name(first_name, last_name) or login

    first_name_en = staff.first_name_en or name_translit(first_name)
    last_name_en = staff.last_name_en or name_translit(last_name)
    name_en = _clear_name(first_name_en, last_name_en) or login

    return EnRuField(ru=name, en=name_en)


def create_staff_field(name: EnRuField, login: str) -> EnRuField:
    if settings.IS_BUSINESS:
        return name

    staff_link = '[{name}](https://staff.yandex-team.ru/{login})'
    return EnRuField(
        en=staff_link.format(name=name.en, login=login),
        ru=staff_link.format(name=name.ru, login=login),
    )


def create_declination_verb(staff: Staff) -> str:
    return 'a' if staff.gender == GENDER.FEMALE else ''


def create_inflect_field(staff: Staff, forms: list[str]) -> UserInflectField:
    translation.activate('ru')
    inflects = {form: staff.inflections.inflect(form) for form in forms}
    translation.deactivate()
    return UserInflectField.parse_obj(inflects)


def _clear_name(first_name, last_name) -> str | None:
    """EBI-1004"""
    parts = [section for section in [first_name, last_name] if '@' not in section]
    return ' '.join(parts) if parts else None
