"""
Функции для работы с правами доступа к страницам вики.

Кстати, права доступа кешируются.
Следует обратить внимание на функции interpret_raw_access, get_bulk_raw_access и has_access.
Они составляют основу работу с правами.


"""

import logging
from copy import deepcopy

from django.conf import settings

from wiki.org import get_org
from wiki.pages.access.raw import get_bulk_raw_access
from wiki.pages.models import Page
from wiki.utils.supertag import translit

logger = logging.getLogger('django.request')

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)

interpreted_access_stub = r = {
    # Доступ по-умолчанию и никак не задан.
    'is_common_wiki': False,
    # Доступ унаследован от родителя
    'is_inherited': False,
    # Доступ по-умолчанию
    'is_common': False,
    # Доступ только авторам страницы
    'is_owner': False,
    # Доступ ограниченный
    'is_restricted': False,
    # Доступ анонимным пользователям
    'is_anonymous': False,
    # Следующие группы имеют доступ, непусто только если доступ ограниченный. Обратное неверно.
    'groups': [],
    # Следующие пользователи (модель Staff) имеют доступ, непусто только если доступ ограниченный. Обратное неверно.
    'users': [],
    'pending_groups': [],
    'pending_users': [],
}


def fill_group_and_users(list, result):
    """Разложить права из списка в словарь

    Хелпер для interpret_raw_access
    """
    for a in list:
        if a.group:
            result['groups'].append(a.group)
        elif a.staff:
            result['users'].append(a.staff)


def interpret_raw_access(raw_access):
    """
    Дать полную картину прав доступа к странице

    Результат работы - dict c очищенными данными о правах доступа к странице.
    шаблон interpreted_access_stub прокомментирован вверху модуля.

    @type raw_access: dict
    @param raw_access: словарь с правами доступа для отдельной страницы, в формате, возвращаемом get_bulk_raw_access

    @rtype: dict
    @return: очищенные данные о правах доступа к странице
    """
    r = deepcopy(interpreted_access_stub)
    if raw_access['list']:
        if raw_access['latest_supertag'] != raw_access['supertag']:
            r['is_inherited'] = True

        if raw_access['list_len'] == 1:
            if raw_access['list'][0].is_common:
                r['is_common'] = True
                return r
            elif raw_access['list'][0].is_owner:
                r['is_owner'] = True
                return r
            elif raw_access['list'][0].is_anonymous:
                r['is_anonymous'] = True
                return r
        r['is_restricted'] = True
        fill_group_and_users(raw_access['list'], r)
    else:
        r['is_common_wiki'] = True
    return r


def get_inheritors(supertag):
    """
    Вернуть список супертегов детей, которые наследуют доступ.
    В случае, когда страницы с таким supertag нет -- кидает Page.DoesNotExist

    @param supertag: строка

    string -> [supertag, supertag, ... ]

    """
    result = [Page.objects.get(supertag=supertag, org=get_org()).supertag]

    # функция-помощник
    in_level = lambda supertag: supertag.count('/') == level

    children = (
        Page.objects.filter(supertag__startswith=translit(supertag) + '/', org=get_org())
        .order_by('supertag')
        .values_list('supertag', flat=True)
    )

    children = set(children)
    # чтобы выбрать все дерево
    analyzed_children = set([])

    # что не должно попасть в результат
    skip_branch = []
    level = supertag.count('/')

    while True:
        level += 1
        children_on_level = list(filter(in_level, children))
        analyzed_children.update(children_on_level)
        # Отфильтровать ненужные узлы.
        for child in children_on_level:
            if any(child.startswith(supertag + '/') for supertag in skip_branch):
                children_on_level.remove(child)

        raw_access = get_bulk_raw_access(Page.objects.filter(supertag__in=children_on_level, org=get_org()))

        for child in raw_access:
            if interpret_raw_access(raw_access[child])['is_inherited']:
                result.append(child.supertag)
            else:
                skip_branch.append(child.supertag)

        if analyzed_children == children:
            break

    return result


def get_pages_accessible_to_staff_or_group(pages, staff_ids, group_ids):
    """
    Генератор, возвращает те из переданных страниц, которые доступны переданным пользователям или группам

    @param pages: список объектов Page
    @param staff_ids: список id объектов Staff
    @param group_ids: список id объектов Group

    [Page, Page, ...], [Staff.id, Staff.id, ... ], [Group.id, Group.id, ... ] -> [Page, ... ]
    """
    access = get_bulk_raw_access(pages)

    for page in access:
        # парсим права в удобный вид
        page_access = interpret_raw_access(access[page])

        # проверяем, доступна ли страница нужным пользователям и группам
        accessible_for_users = any((user.id in staff_ids) for user in page_access['users'])
        accessible_for_groups = any((group.id in group_ids) for group in page_access['groups'])

        if accessible_for_users or accessible_for_groups:
            yield page
