"""
Функции для получения статуса доступа в терминах ACCESS_RESTRICTED, ACCESS_COMMON итп.

@author: chapson
"""

from logging import getLogger
from typing import List, Dict

from django.conf import settings
from wiki.intranet.models import Staff

from wiki.org import get_org, org_staff
from wiki.pages.access.cache import get_cached_access_status, set_cached_access_status
from wiki.pages.access.groups import user_group_ids
from wiki.pages.access.raw import get_bulk_raw_access, get_raw_access
from wiki.pages.logic import hierarchy
from wiki.pages.models import Access, Page
from wiki.utils.supertag import tag_to_supertag, translit

log = getLogger(__name__)

ACCESS_NONEXISTENT = -1
ACCESS_DENIED = 0
ACCESS_COMMON = 1
ACCESS_RESTRICTED = 2
ACCESS_UNLIMITED = 3

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)


def _get_page(supertag):
    try:
        return Page.active.prefetch_related('authors').only('authors').get(supertag=supertag, org=get_org())
    except Page.DoesNotExist:
        return None


def current_org_id():
    org = get_org()
    if org is None:
        return org
    return org.id


def _get_page_cached(latest_supertag, page_cache=None):
    if page_cache is None:  # bypass cache
        return _get_page(latest_supertag)

    cache_hit = latest_supertag in page_cache
    parent_page = page_cache.get(latest_supertag)

    if cache_hit and parent_page is not None and parent_page.org_id != current_org_id():
        # мы не промазали (закешированным может быть и None), однако в кеше странца от другой организации
        # Этот кейс не должен случиться, так как контекст организации в рамках одной проверки не
        # не изменяется. Но на всякий случай я обязан это добавить.
        cache_hit = False
        log.error('Page cache was poisoned:' + f'Current Org ID: {current_org_id()}, cached page {parent_page.org_id}')

    if not cache_hit:
        parent_page = _get_page(latest_supertag)
        page_cache[latest_supertag] = parent_page

    return parent_page


def _get_page_authors_with_inherited(current_page, raw_access, page_cache=None):
    authors = list(current_page.get_authors())
    if raw_access['latest_supertag'] and raw_access['latest_supertag'] != current_page.supertag:
        # current_page наследует права от родительской страницы.
        # В этом случае необходимо пронаследовать авторов родительской страницы:
        # добавим в авторы страницы всех авторов родительской страницы

        latest_supertag = raw_access['latest_supertag']
        parent_page = _get_page_cached(latest_supertag, page_cache=page_cache)

        if parent_page:
            for author in parent_page.get_authors():
                if author not in authors:
                    authors.append(author)

    return authors


def _has_access(tag, user, privilege='read', current_page=None):
    # Функция очень похожа на bulk_has_access, но у них немного разные
    # сигнатуры - у той есть access_list, который нужно формировать снаружи.
    # Явно должен остаться только один.
    if getattr(settings, 'read_only', False) and (privilege == 'write' or privilege == 'comment'):
        return ACCESS_DENIED
    if getattr(settings, 'ignore_acls', False):
        return ACCESS_UNLIMITED

    if not current_page:
        supertag = tag_to_supertag(tag)
        current_page = _get_page(supertag)
        if not current_page:
            parent = hierarchy.get_nearest_existing_parent(supertag)
            if parent is None:
                return ACCESS_COMMON
            return _has_access(parent.supertag, user, privilege, parent)

    raw_access = get_raw_access(current_page)
    authors = _get_page_authors_with_inherited(current_page, raw_access)

    if user in authors:
        if raw_access['list_len'] > 0:
            if raw_access['list'][0].is_common:
                return ACCESS_COMMON
            return ACCESS_RESTRICTED
        else:
            return ACCESS_COMMON

    # is_common
    if not (raw_access['list_len']):
        return has_common_access(user, privilege)

    # is_common or is_owner
    if raw_access['list_len'] == 1:
        if raw_access['list'][0].is_common:
            return has_common_access(user, privilege)
        elif raw_access['list'][0].is_anonymous:
            if privilege == 'read':
                return ACCESS_COMMON
            elif privilege == 'write' and user.is_authenticated:
                return ACCESS_COMMON
        elif raw_access['list'][0].is_owner:
            if raw_access['list'][0].page.supertag == raw_access['supertag']:
                if user in raw_access['list'][0].page.get_authors():
                    return ACCESS_RESTRICTED
                else:
                    return ACCESS_DENIED
            else:
                try:
                    if user.is_authenticated and user in authors:
                        return ACCESS_RESTRICTED
                    else:
                        return ACCESS_DENIED
                except Page.DoesNotExist:
                    pass

    if not user.is_authenticated:
        return ACCESS_DENIED

    # check for the owner
    try:
        current_page = (
            Page.objects.prefetch_related('authors').only('authors').get(supertag=raw_access['supertag'], org=get_org())
        )
        if user in current_page.authors.all():
            return ACCESS_RESTRICTED
    except Page.DoesNotExist:
        pass

    try:
        staff = org_staff().get(user=user)
    except Staff.DoesNotExist:
        staff = None

    if staff:
        group_ids = user_group_ids(staff)
    else:
        group_ids = []

    for ac in raw_access['list']:
        if staff and ac.staff and ac.staff == staff:
            return ACCESS_RESTRICTED
        elif group_ids and ac.group and ac.group.id in group_ids:
            return ACCESS_RESTRICTED
    return ACCESS_DENIED


def bulk_has_access(page, user, raw_access_list, privilege='read', page_cache=None):
    if getattr(settings, 'read_only', False) and (privilege == 'write' or privilege == 'comment'):
        return ACCESS_DENIED
    if getattr(settings, 'ignore_acls', False):
        return ACCESS_UNLIMITED

    if not user.is_authenticated:
        return ACCESS_DENIED

    raw_access = raw_access_list.get(page)
    if raw_access is None:
        # TODO: понять, как можно избавиться от этого вызова
        raw_access = get_raw_access(page)

    authors = _get_page_authors_with_inherited(page, raw_access, page_cache)

    if user in authors:
        if raw_access['list_len'] > 0:
            if raw_access['list'][0].is_common:
                return ACCESS_COMMON
            elif raw_access['list'][0].is_anonymous and privilege == 'read':
                return ACCESS_COMMON
            return ACCESS_RESTRICTED
        else:
            return ACCESS_COMMON

    # is_common
    if not raw_access['list_len']:
        return has_common_access(user, privilege)

    # is_common or is_owner
    if raw_access['list_len'] == 1 and isinstance(raw_access['list'][0], Access):
        if raw_access['list'][0].is_common:
            return has_common_access(user, privilege)
        elif raw_access['list'][0].is_anonymous and privilege == 'read':
            return ACCESS_COMMON
        elif raw_access['list'][0].is_owner:
            if raw_access['list'][0].page.supertag == raw_access['supertag']:
                if user in raw_access['list'][0].page.authors.all():
                    return ACCESS_RESTRICTED
                else:
                    return ACCESS_DENIED
            else:
                if user.is_authenticated and user in page.authors.all():
                    return ACCESS_RESTRICTED
                else:
                    return ACCESS_DENIED

    # check for the owner
    if user in page.authors.all():
        return ACCESS_RESTRICTED

    staff = user.staff

    if staff:
        group_ids = user_group_ids(staff)
    else:
        group_ids = []

    for ac in raw_access['list']:
        if staff and ac.staff and ac.staff == staff:
            return ACCESS_RESTRICTED
        elif group_ids and ac.group and ac.group.id in group_ids:
            return ACCESS_RESTRICTED

    return ACCESS_DENIED


def get_bulk_access_status(tags, user):
    """Получить статус прав доступа (чтения) к вики-страницам в терминах
    ACCESS_UNLIMITED, ACCESS_DENIED итп.

    По данному списку супертегов и джанго пользователю возвращает словарь с правами доступа.
    Функция замещает собой функцию wiki.pages.access.get_access_status

    @param tags: tags of pages

    @param user: django user
    """
    # Эта функция - только про чтение. A эта переменная здесь как напоминание об
    # этом. Нельзя просто так взять и поменять значение этой переменной,
    # работать не будет.
    privilege = 'read'

    result = {}
    missing = set()

    # .explainlinks передаёт в эту функцию тэги, в остальных местах сюда
    # передаются именно супертэги.
    # TODO: отрефаторить. чтобы все передавали супертэги, и убрать приведение.

    supertag_to_tag_map = {}

    for tag in tags:
        supertag = translit(tag)
        access_status = get_cached_access_status(supertag, user.id, privilege)
        if access_status is None:
            missing.add(supertag)
            supertag_to_tag_map[supertag] = tag
        else:
            result[tag] = access_status

    pages = list(
        Page.objects.filter(supertag__in=missing, org=get_org())
        .prefetch_related('authors')
        .only('supertag', 'tag', 'authors')
    )

    # get current access rights for pages and ancestors
    raw_access_list = get_bulk_raw_access(pages)

    supertag_to_page_map = {p.supertag: p for p in pages}
    page_cache = supertag_to_page_map.copy()

    for supertag in missing:

        page = supertag_to_page_map.get(supertag)
        if not page:
            access_status = ACCESS_NONEXISTENT
        else:
            access_status = bulk_has_access(page, user, raw_access_list, page_cache=page_cache)
        tag = supertag_to_tag_map[supertag]
        result[tag] = access_status
        set_cached_access_status(supertag, user.id, access_status, privilege)

    return result


def get_bulk_access_status_of_pages(pages: List[Page], user) -> Dict[str, bool]:
    """
    Рекомендуется в pages:
    .prefetch_related('authors')
    .only('supertag', 'tag', 'authors')
    """
    privilege = 'read'
    result = {}
    raw_access_list = get_bulk_raw_access(pages)
    page_cache = {p.supertag: p for p in pages}

    for page in pages:
        access_status = bulk_has_access(page, user, raw_access_list, page_cache=page_cache)
        result[page.slug] = access_status
        set_cached_access_status(page.supertag, user.id, access_status, privilege)

    return result


def get_access_status(tag, user):
    """Получить доступ в терминах ACCESS_NONEXISTENT, ACCESS_DENIED итп"""
    return get_bulk_access_status([tag], user).get(tag, ACCESS_NONEXISTENT)


if IS_INTRANET:
    from wiki.pages.access.groups import common_access_is_ok

    def has_common_access(user, privilege='read'):
        if user.is_authenticated:
            try:
                staff = user.staff
            except Staff.DoesNotExist:
                return ACCESS_DENIED

            # staff can be None with intranet auth middleware
            # пользуемся кешированным результатом
            if staff is not None and common_access_is_ok(staff):
                return ACCESS_COMMON
        return ACCESS_DENIED

else:

    def has_common_access(user, privilege='read'):
        if user.is_authenticated:
            return ACCESS_COMMON
        return ACCESS_DENIED


def has_access(tag, user, privilege='read', current_page=None):
    """Ответить truе, если у пользователя есть права редактировать данную страницу

    Если не хотите передавать tag, используйте параметр current_page.

    @param tag: тег страницы
    @param user: Django User
    @param privilege: привилегия (не используется)
    @param current_page: страница
    """

    supertag = translit(tag)
    cache_result = get_cached_access_status(supertag, user.id, privilege)
    if cache_result is not None:
        return bool(cache_result)

    access_result = _has_access(tag, user, privilege, current_page)
    set_cached_access_status(supertag, user.id, access_result, privilege)
    return bool(access_result)


def get_cluster_accessible_pages(supertag, user):
    """
    Возвращает доступные для пользователя страницы в кластере
    """
    queryset = Page.active.get_descendants(supertag, True)
    queryset = queryset.only('id', 'supertag')
    pages = list(queryset)
    supertags = [page.supertag for page in pages]
    access_status = get_bulk_access_status(supertags, user)
    accessible_pages = []
    for page in pages:
        if access_status[page.supertag] > 0:
            accessible_pages.append(page)
    return accessible_pages


def _page_has_access(self, user, privilege='read'):
    return has_access(self.supertag, user, privilege, current_page=self)


Page.has_access = _page_has_access
