"""
Функции для проверки прав, связанные с работой с кешом.

@author: chapson
"""

import uuid
from typing import List

from django.conf import settings
from django.core.cache import caches

from wiki.org import get_org_id
from wiki.utils.per_request_memoizer import memoize_for_this_request

# 10 minute caches should be enough for anyone!
ACCESS_STATUS_TIMEOUT = 10 * 60
GROUPS_TIMEOUT = 10 * 60

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)

CACHE_ACL_BACKEND = caches['access_rights']
CACHE_USER_GROUPS_BACKEND = caches['user_groups']


def cache_is_enabled():
    """
    @rtype: bool
    """
    return True


def supertag_token_key(supertag):
    return 'SUPERTAG_TOKEN:%s:%d' % (supertag, get_org_id() or 0)


def supertag_access_cache_key(supertag, user_id, privilege):
    """
    Ключ в кэше для прав юзера на супертэг
    """
    # Токен нужен для инвалидации прав на супертэги (сразу для всех юзеров).
    token_key = supertag_token_key(supertag)
    token = CACHE_ACL_BACKEND.get(token_key)
    if not token:
        token = str(uuid.uuid4())
        CACHE_ACL_BACKEND.set(token_key, token, 0)

    return 'SUPERTAG_ACCESS:%s:%d:%s:%d' % (token, user_id, privilege, get_org_id() or 0)


def get_cached_access_status(supertag, user_id, privilege):
    if not cache_is_enabled() or not user_id:
        # user_id = None happens in tests, lets just ignore it
        return None
    return CACHE_ACL_BACKEND.get(supertag_access_cache_key(supertag, user_id, privilege))


def set_cached_access_status(supertag, user_id, access_status, privilege):
    # access_status - это можно или нет (одна из констант ACCESS_*), а privilege
    # - это что: чтение, запись и т. д.
    if not cache_is_enabled() or not user_id:
        # user_id = None happens in tests, lets just ignore it
        return
    CACHE_ACL_BACKEND.set(supertag_access_cache_key(supertag, user_id, privilege), access_status, ACCESS_STATUS_TIMEOUT)


def invalidate_access_rights(supertags):
    CACHE_ACL_BACKEND.delete_many(keys=(supertag_token_key(s) for s in supertags))


def all_groups_cached_key(staff):
    return 'all_groups_cached_{0}_{1}'.format(staff.login, get_org_id() or 0)


@memoize_for_this_request
def all_group_ids_cached(staff) -> List[int]:
    """Получить список id групп пользователя (с кэшированием)

    Пожалуйста, используйте обертку wiki.pages.access.user_group_ids

    Staff -> []

    """
    key = all_groups_cached_key(staff)
    result = None
    if cache_is_enabled():
        result = CACHE_USER_GROUPS_BACKEND.get(key)
    if result is None:
        result = staff.user.get_all_group_ids()
        if cache_is_enabled():
            CACHE_USER_GROUPS_BACKEND.set(key, result, GROUPS_TIMEOUT)

    return result
