import time

from django.db.models.query import QuerySet  # noqa
from wiki.intranet.models import Staff

from wiki.pages.access.access import get_inheritors, get_pages_accessible_to_staff_or_group, interpret_raw_access
from wiki.pages.access.raw import get_bulk_raw_access
from wiki.pages.models import Page
from wiki.utils.timer import track_time

ACTIVE = True
try:
    from wiki.intranet.models import Group
except ImportError:
    ACTIVE = False


def _get_external_groups():
    """
    Вернуть группы количеством внешних пользователей больше 0

    @rtype: QuerySet
    @return: QuerySet объектов Group
    """

    return Group.objects.filter(externals_count__gt=0)


def _get_external_staff():
    """
    Вернуть внешних пользователей.
    Внешние пользователи состоят департаменте с url, начинаюшимся на yandex

    @rtype: QuerySet
    @return: QuerySet объектов Staff
    """

    return (
        Staff.objects.select_related('department')
        .filter(is_dismissed=False)
        .exclude(department__url__startswith='yandex')
    )


def update_ext_access_status(supertag):
    """
    Обновить флаг opened_to_external_flag страницы и подстраниц, наследующих доступ.

    ВНИМАНИЕ: Пользователи, которые не входят ни в одну группу не считаются внешними!

    @type supertag: string
    @param supertag: supertag страницы

    """
    if not ACTIVE:
        raise NotImplementedError('No groups in intranet_stuff')

    # супертеги всех страниц, наследующих доступ от переданной
    pages_supertags = get_inheritors(supertag)
    pages = Page.active.filter(supertag__in=pages_supertags)

    bulk_update_ext_access_status(pages)


def _update_ext_access_status(pages, chunk_size):
    """
    @param pages: множество страниц над которым надо работать.
    @param chunk_size: по сколько страниц выбирать за раз (для облегчения запросов).
    """

    # получаем списки внешних пользователей и групп, в которых внешние пользователи есть
    staff = _get_external_staff().values_list('id', flat=True)
    groups = _get_external_groups().values_list('id', flat=True)

    n = 0
    last_pk = -1
    while True:
        # тут выбираем объекты page, потому что проверка прав доступа в
        # get_pages_accessible_to_staff_or_group требует объект page.
        # Это не очень хорошо - много памяти нужно.

        if pages is None:
            pages_qs = Page.active.all()
        else:
            pages_qs = pages

        pages_chunk = list(pages_qs.filter(id__gt=last_pk).order_by('id')[:chunk_size])

        if not len(pages_chunk):
            break

        last_pk = pages_chunk[-1].id

        n += chunk_size

        # проверяем, какие из внешних страниц доступны внешним
        opened_pages = list(get_pages_accessible_to_staff_or_group(pages_chunk, staff, groups))

        if opened_pages:
            page_ids_to_mark_open = [page.id for page in opened_pages if not page.opened_to_external_flag]

            if page_ids_to_mark_open:
                # кому надо выставляется opened_to_external_flag True
                Page.objects.filter(id__in=page_ids_to_mark_open).update(opened_to_external_flag=True)

        opened_page_ids = [page.id for page in opened_pages]

        page_ids_to_unmark_open = [
            page.id for page in pages_chunk if page.opened_to_external_flag and page.id not in opened_page_ids
        ]

        if page_ids_to_unmark_open:
            # кому надо выставляется opened_to_external_flag False
            Page.objects.filter(id__in=page_ids_to_unmark_open).update(opened_to_external_flag=False)


@track_time
def bulk_update_ext_access_status(pages=None, chunk_limit=500):
    """
    Обновить поле opened_to_external_flag (доступна ли страница внешним пользователям)
    у всех переданных страниц.

    ВНИМАНИЕ: Пользователи, которые не входят ни в одну группу не считаются внешними!

    @type pages: QuerySet
    @param pages: страницы, у которых необходимо обновить поле opened_to_external_flag
    @param chunk_limit:  размер куска списка страниц, с которыми будем работать.
    """
    if not ACTIVE:
        raise NotImplementedError('No groups in intranet_stuff')
    _update_ext_access_status(pages, chunk_limit)


def is_available_to_ext_users(page):
    """
    Возвращает True, если переданная страница доступна для внешних
    пользователей или групп, внешних пользователей содержащих.

    ВНИМАНИЕ! Это -- динамический способ узнать, доступна ли страница. У модели Page есть
    поле opened_to_external_flag, но оно пока обновляется по крону

    @type page: Page
    @rtype: bool
    """

    # структура с правами для страницы и ее предков
    raw_access_list = get_bulk_raw_access([page])
    # действующие для страницы разрешения
    page_access = interpret_raw_access(raw_access_list[page])

    if page_access['is_restricted']:
        return any(u.is_external_employee for u in page_access['users']) or any(
            g.externals_count > 0 for g in page_access['groups']
        )

    return False


def _interpret_access_for_all_pages(page, interpreted_access):
    if interpreted_access['is_owner']:
        if page.get_authors():
            try:
                return page, [author.staff for author in page.get_authors()], []
            except Staff.DoesNotExist:
                # TODO: WIKI-7099 - удалить в рамках этого тикета
                return page, [], []
        else:
            return page, [], []

    elif interpreted_access['is_restricted']:
        return page, interpreted_access['users'], interpreted_access['groups']
    # анонимного доступа в основном инстансе нет, не реализуем
    elif interpreted_access['is_anonymous']:
        raise ValueError('Not implemented')
    elif interpreted_access['is_common'] or interpreted_access['is_common_wiki']:  # common access
        return page, None, None
    return page, None, None


def _interpret_external_access_for_all_pages(page, interpreted_access):
    if interpreted_access['is_restricted']:
        # если разрешения отличны от дефолтных

        # не уволенные внешние пользователи с доступом
        ext_users = [u for u in interpreted_access['users'] if u.is_external_employee and u.is_active]
        # группы с внешними пользователями, у которых есть доступ
        groups_with_ext = [g for g in interpreted_access['groups'] if g.externals_count > 0]

        # возвращаются страница, внешние пользователи и группы
        # с внешними пользователями
        return page, ext_users, groups_with_ext
    return page, None, None


def current_timestamp():
    """
    Обертка для удобного тестирования.
    """
    return time.time()


def all_active_pages_query_set():
    """
    Обертка для удобного тестирования.
    """
    return Page.active.all()


def format_for_security(page_users_groups_gen):
    """
    Отформатировать данные так, как нужно безопасникам.
    """
    result = []
    for page, users, groups in page_users_groups_gen:
        result.append(
            {
                'uri': page.supertag,
                'title': page.title,
                'users': None if users is None else [u.uid for u in users],
                'groups': None if groups is None else [g.url for g in groups],
            }
        )

    return {'timestamp': current_timestamp(), 'pages': result}
